#pragma once

#include <enyx/cores_c/ethernet_filter/ethernet_filter.h>

#include <enyx/cores/result.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/hw/core.hpp>
#include <memory>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_ethernet_filter>
{
    void
    operator()(::enyx_ethernet_filter * ptr) const
    {
        ::enyx_ethernet_filter_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace ethernet_filter {

class ethernet_filter
{
public:
    /**
     * Construct the ETHERNET FILTER module
     *
     * This module configures the following hw::core:
     *
     * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     *
     * @param ethernet_filter_core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    ethernet_filter(enyx::hw::core const& ethernet_filter_core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Check is the filter is enabled.
     *
     * @return The filter activation status
     */
    bool
    enabled() const noexcept;

    /**
     * Enable filter.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable() noexcept;

    /**
     * Disable filter.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable() noexcept;

    /**
     * Check if unicast promiscuous mode is enabled.
     * When enabled, do not check Unicast MAC, forward all packets.
     *
     * Default disabled.
     *
     * @return The unicast promiscuous activation status
     */
    bool
    unicast_promiscuous_enabled() const noexcept;

    /**
     * Enable unicast promiscuous mode.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_unicast_promiscuous() noexcept;

    /**
     * Disable unicast promiscuous mode.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_unicast_promiscuous() noexcept;

    /**
     * Check if UDP session copy to filter is enabled.
     *
     * When enabled, all packets from this UDP session will be copied to filter.
     *
     * @param udp_session_id The UDP session to check
     * @return A result object containing the copy to filter activation status
     * or an error on failure.
     */
    result<bool>
    udp_session_copy_enabled(std::uint16_t udp_session_id) const noexcept;

    /**
     * Enable UDP session copy to filter.
     *
     * @param udp_session_id The UDP session to enable.
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_udp_session_copy(std::uint16_t udp_session_id) noexcept;

    /**
     * Disable UDP session copy to filter.
     *
     * @param udp_session_id The UDP session to disable.
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_udp_session_copy(std::uint16_t udp_session_id) noexcept;

    /**
     * Get the number of packet before applying the filter.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    enyx::result<std::uint32_t>
    get_packet_fifo_out_count() const noexcept;

    /**
     * Get the number of packet dropped by the filter.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    enyx::result<std::uint32_t>
    get_packet_filter_drop_count() const noexcept;

    /**
     * Get the number of packets after applying the filter.
     *
     * @return A result object containing the packet count on success or an
     * error on failure
     */
    enyx::result<std::uint32_t>
    get_packet_filter_out_count() const noexcept;

    /**
     * Get the filtered data width.
     *
     * @return A result object containing the filtered data width on success
     * or an error on failure
     */
    enyx::result<std::uint8_t>
    get_filtered_data_width() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_ethernet_filter *
    handle() const noexcept;

private:
    enyx::hw::core ethernet_filter_core_;
    std::unique_ptr<::enyx_ethernet_filter> ethernet_filter_;
};

} /* namespace ethernet_filter */
ENYX_CORES_NAMESPACE_END

#include "ethernet_filter.ipp"
