#pragma once

#include <memory>

#include <enyx/cores_c/data_stream/source.h>

#include <enyx/cores/namespace.hpp>

ENYX_CORES_NAMESPACE_BEGIN

namespace data_stream {

/**
 *  Interface for the @ref source wrapper encapsulating its templated type.
 */
class source
{
public:
    /**
     *  Construct an unitialized source
     */
    source();

    /**
     *  Construct a cpp source from a c_source handle
     */
    source(::enyx_data_stream_source const& c_source);

    /**
     *  Destructor
     */
    virtual
    ~source() = default;

    /**
     * Retrieve the @b C handle associated with this source
     *
     * @return The C handle
     */
    ::enyx_data_stream_source const *
    handle() const noexcept;

    /**
     * Retrieve the @b C handle associated with this source
     *
     * @return The C handle
     */
    ::enyx_data_stream_source *
    handle() noexcept;

private:
    ::enyx_data_stream_source c_source_;
};

/**
 *  C++ wrapper around @b C @ref enyx_data_stream_source without any specific
 *  implementation.
 *
 *  It is a basic type that can be used to implement custom source types.
 *
 *  @tparam Callable The @p callable type
 *
 *  Usage:
 *  @code
 *  // Create the data handler.
 *  // It could be a stateful functor with states as well.
 *  auto on_data = [](std::uint8_t const * data,
 *                    std::uint32_t size,
 *                    void const * metadata) {
 *      std::cout << "Received " << size << " bytes-long message" << std::endl;
 *  };
 *
 *  // Create the C++ source
 *  enyx::data_stream::source cxx_source{on_data};
 *
 *  // Create the HW source
 *  enyx::data_stream::hw_source hw_source{a2c_stream, cxx_source};
 *
 *  // Poll for messages until exit is requested or an error occurs.
 *  while (! is_exit_requested)
 *    if (hw_source.poll_once() < 0)
 *        throw std::system_error{errno, std::generic_category(),
 *                                "hw_source::poll_once"};
 *  @endcode
 */
template<typename Callable>
class functor_source final : public source
{
public:
    /**
     * Create an data_stream source calling @p callable on data
     *
     * @param callable The callable called on data
     */
    functor_source(Callable&& callable);

    /**
     * Deleted copy-constructor (and thus move-constructor).
     *
     * To prevent copy and move of source objects because
     * this class instances cannot be moved as their addresses are captured
     * and given to the underlying C source.
     */
    functor_source(functor_source const&) = delete;

    /**
     * Deleted copy assignment operator too.
     *
     * See copy-constructor documentation.
     */
    functor_source &
    operator=(functor_source const&) = delete;

private:
    static void
    on_data(std::uint8_t const * data,
            std::uint32_t size,
            void const * metadata,
            void * opaque);

private:
    Callable callable_;
};

/// Represents a source pointer
using source_ptr = std::unique_ptr<source>;

/**
 *  Create a source ptr from a provided @p handler
 *
 *  @note This function will either create a source or a
 *  functor_source (source derived) depending on the Handler type.
 *
 *  @tparam Handler The provided handler type
 *  @param handler The handler to use
 *  @return The created source
 */
template<typename Handler>
source_ptr
make_source(Handler && handler);

} /* namespace data_stream */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/data_stream/source.ipp>
