#pragma once

#include <enyx/cores_c/data_stream/sink.h>

#include <cerrno>
#include <memory>
#include <system_error>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores/data_stream/metadata.hpp>

ENYX_CORES_NAMESPACE_BEGIN
namespace data_stream {

/**
 *  C++ wrapper around @b C @ref enyx_data_stream_sink
 *
 *  Usage:
 *  @code
 *  ::enyx_data_stream_sink c_sink{};
 *  // Initialize the c_sink
 *  // ...
 *
 *  // Create the C++ sink
 *  enyx::data_stream::sink cxx_sink{c_sink};
 *
 *  // Send a buffer containing 'test'
 *  std::array<std::uint8_t, 4> buffer{'t', 'e', 's', 't'};
 *  int failure;
 *  do
 *      failure = cxx_sink.send(buffer.data(), buffer.size(), nullptr);
 *  while (failure < 0 && errno == EAGAIN);
 *
 *  // Or throw an exception if the sending failed
 *  if (failure)
 *      throw std::system_error{errno, std::generic_category(),
 *                              "enyx::data_stream::sink::send"};
 *  @endcode
 */
class sink final
{
public:
    /// @copydoc enyx_data_stream_sink_buffer
    using buffer = ::enyx_data_stream_sink_buffer;

public:
    /**
     *  Create a data_stream sink from a user filled @p sink
     *
     *  @param sink A data_stream  sink filled by the user
     */
    explicit
    sink(::enyx_data_stream_sink const& sink) noexcept;

    /**
     * Send @p vector_count @p vectors of data to the accelerator in a row.
     *
     * @param buffer Pointer to the head of the buffer queue.
     * @param metadata The message metadata chain
     * @return 0 on success, -1 on error (@b errno is set accordingly)
     *
     * @note When the accelerator fails to keep up with the bandwidth, @b errno
     * will be set to @b EAGAIN. In that case, you may call send again
     */
    int
    send(buffer const * buffer,
         void const * metadata) noexcept;

    /**
     * Send @p size byte(s) from @p data to the accelerator.
     *
     * @param data Pointer to the data to send.
     * @param size The byte(s) count to send.
     * @param metadata The message metadata chain
     * @return 0 on success, -1 on error (@b errno is set accordingly)
     *
     * @note When the accelerator can't keep up with the bandwidth, @b errno
     * will be set to @b EAGAIN. In that case, you may call send again.
     */
    int
    send(void const * data,
         std::uint32_t size,
         void const * metadata) noexcept;

    /**
     * Get sink MTU.
     *
     * @return A result object containg MTU on success or an error on failure.
     */
    result<size_t>
    get_mtu() const noexcept;

    /**
     * Retrieve the @b C handle associated with this sink
     *
     * @return The C handle
     */
    ::enyx_data_stream_sink *
    handle() const noexcept;

private:
    std::unique_ptr<::enyx_data_stream_sink> sink_;
};

} /* namespace data_stream */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/data_stream/sink.ipp>
