#pragma once

#include <enyx/hw_c/mock.h>
#include <enyx/hw_c/mocking/mmio.h>
#include <enyx/hw_c/mocking/a2c_stream.h>
#include <enyx/hw_c/mocking/c2a_stream.h>
#include <enyx/hw_c/product.h>

/**
 * This struct represents a mocked accelerator.
 */
typedef struct enyx_hwm_accelerator enyx_hwm_accelerator;

/**
 * This type represents the accelerator reset callback type.
 */
typedef void (*enyx_hwm_accelerator_reset_cb_t)(void * context);

/**
 * Create a mocked accelerator and install it on a mock.
 *
 * @param mock: The mock to use
 * @return The new mocked accelerator on success or NULL on failure (with @b
 * errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_accelerator *
enyx_hwm_accelerator_create(enyx_hw_mock * mock);

/**
 * Destroy a mocked accelerator.
 *
 * @param accelerator: The mocked accelerator to destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hwm_accelerator_destroy(enyx_hwm_accelerator * accelerator);

/**
 * Set the @p version of a mocked @p accelerator.
 *
 * @param accelerator The accelerator to be modified
 * @param version The version to set
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_set_version(enyx_hwm_accelerator * accelerator,
                                 enyx_hw_product_version const * version);

/**
 * Set the @p name of a mocked @p accelerator.
 *
 * @note The current maximum name size is 256 (including terminating NULL
 * bytes).
 *
 * @param accelerator The accelerator to be modified
 * @param name The name to set
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_set_name(enyx_hwm_accelerator * accelerator,
                                 char const * name);

/**
 * Set the user count of a mocked @p accelerator.
 *
 * @param accelerator The accelerator to be modified
 * @param count The count to set
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_set_user_count(enyx_hwm_accelerator * accelerator,
                                    size_t count);

/**
 * Set the reset callback of a mocked @p accelerator.
 *
 * @param accelerator The accelerator to be modified
 * @param cb The callback to set
 * @param context A user defined context to be used in the callback
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_set_reset_callback(enyx_hwm_accelerator * accelerator,
                                        enyx_hwm_accelerator_reset_cb_t cb,
                                        void * context);

/**
 * Bind a mocked MMIO device interface to a mocked accelerator.
 *
 * @param accelerator: The mocked accelerator to bind to
 * @param mmio: The mocked MMIO interface to bind
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_bind_mmio(enyx_hwm_accelerator * accelerator,
                               enyx_hwm_mmio_interface * mmio);

/**
 * Bind a mocked A2C stream interface to a mocked accelerator.
 *
 * @param accelerator: The mocked accelerator to bind to
 * @param stream: The mocked stream interface to bind
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_bind_a2c_stream(enyx_hwm_accelerator * accelerator,
                                     enyx_hwm_a2c_stream_interface * stream);

/**
 * Bind a mocked C2A stream interface to a mocked accelerator.
 *
 * @param accelerator: The mocked accelerator to bind to
 * @param stream: The mocked stream interface to bind
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_accelerator_bind_c2a_stream(enyx_hwm_accelerator * accelerator,
                                     enyx_hwm_c2a_stream_interface * stream);
