/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator device.
 */
#pragma once

#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/properties.h>
#include <enyx/hw_c/filter.h>
#include <enyx/hw_c/fwd.h>
#include <enyx/hw_c/mmio.h>
#include <enyx/hw_c/c2a_stream.h>
#include <enyx/hw_c/a2c_stream.h>
#include <enyx/hw_c/product.h>

/**
 * This class references an accelerator
 *
 * It is used to construct a accelerator, but doesn't
 * constitute ownership of it.
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_accelerator_descriptor enyx_hw_accelerator_descriptor;

/**
 * Retrieve the properties associated with an accelerator @p descriptor
 *
 * @param descriptor The accelerator descriptor object to query
 * @returns A pointer to the properties object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_properties const *
enyx_hw_accelerator_get_properties(enyx_hw_accelerator_descriptor const * descriptor);

/**
 * Retrieve the version from an accelerator @p descriptor.
 *
 * @param descriptor The accelerator descriptor object to query
 * @param[out] version The version to fill
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_accelerator_get_version(enyx_hw_accelerator_descriptor const * descriptor,
                                enyx_hw_product_version * version);

/**
 * Retrieve the accelerator's user count.
 *
 * It can be used to check if the accelerator is used before resetting it.
 *
 * @param descriptor The accelerator descriptor object to query
 * @return The user count on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_accelerator_count_users(enyx_hw_accelerator_descriptor const * descriptor);

/**
 * Reset the accelerator.
 *
 * The accelerator must not be in use to trigger a reset. Resetting it will
 * make the FPGA to reload its firmware from the flash memory.
 *
 * @param descriptor The accelerator descriptor object to use
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_accelerator_reset(enyx_hw_accelerator_descriptor const * descriptor);

/**
 * Count all accelerators.
 *
 * @note @b errno should be set to zero prior calling this function
 *       in order to check for possible error.
 *
 * @return the number of detected accelerators, 0 on error (@b errno is
 *         set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY size_t
enyx_hw_count_accelerators(void);

/**
 * Retrieve all accelerators regardless of their properties
 *
 * @param [out] descriptors The array filled with the descriptors found
 * @param capacity The array capacity
 * @return 0 on success, -1 on error (@b errno is set accordingly).
 *         Error codes are:
 *            - @b EINVAL if @b descriptors is NULL,
 *            - @b ENOBUFS if @b capacity is too small
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_enumerate_all_accelerators(enyx_hw_accelerator_descriptor const ** descriptors,
                                   size_t capacity);

/**
 * Retrieve all accelerator matching @p filter argument
 *
 * This function is used to request particular accelerator(s).
 * The @p filter argument should be filled with filter
 * unique to the requested accelerator(s).
 *
 * @param filter accelerator filter
 * @param descriptors The array filled with the descriptors found
 * @param capacity The array capacity
 * @return The count of selected accelerators on success, -1 on error
 *         (@b errno is set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_enumerate_accelerators(enyx_hw_filter const * filter,
                               enyx_hw_accelerator_descriptor const ** descriptors,
                               size_t capacity);

/**
 * Free accelerators descriptors array
 *
 * @param descriptors Array filled by @ref enyx_hw_enumerate_all_accelerators or
 *            @ref enyx_hw_enumerate_accelerators
 * @param capacity The array capacity
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_free_accelerator_descriptors(enyx_hw_accelerator_descriptor const ** descriptors,
                              size_t capacity);

/**
 * Get an accelerator from its descriptor.
 *
 * @param descriptor The descriptor corresponding to the accelerator
 * @return The requested accelerator on success, NULL on error with errno set.
 *
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_accelerator *
enyx_hw_accelerator_get(enyx_hw_accelerator_descriptor const * descriptor);

/**
 * Release an accelerator.
 *
 * @param accelerator Accelerator to release.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_accelerator_put(enyx_hw_accelerator * accelerator);

/**
 * Get the descriptor of an accelerator
 *
 * @param accelerator Accelerator to get descriptor from
 * @return The descriptor of the accelerator on success, NULL on failure with errno
 * set.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_accelerator_descriptor const *
enyx_hw_accelerator_get_descriptor(enyx_hw_accelerator const * accelerator);

/**
 * Free a previously getted descriptor.
 *
 * @param descriptor The descriptor to free.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_accelerator_put_descriptor(enyx_hw_accelerator_descriptor const * descriptor);

/**
 * Get the number of mmios for this accelerator.
 * @param accelerator Accelerator to count mmios for
 * @return The number of mmios on success, 0 with errno set on error.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY size_t
enyx_hw_accelerator_count_mmios(enyx_hw_accelerator * accelerator);

/**
 * Enumerate all MMIO descriptors for a given accelerator.
 *
 * @param accelerator Accelerator used.
 * @param [out] descriptors Array of @ref enyx_hw_mmio_descriptor.
 * @param capacity size of descriptors.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_accelerator_enumerate_all_mmios(enyx_hw_accelerator * accelerator,
        enyx_hw_mmio_descriptor const ** descriptors, size_t capacity);

/**
 * Enumerate MMIO descriptors matching filter for a given accelerator.
 *
 * @param accelerator Accelerator used.
 * @param filter filter to check
 * @param [out] descriptors Array of @ref enyx_hw_mmio_descriptor.
 * @param capacity size of descriptors.
 * @return The count of selected mmios on success, -1 with errno set on failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_accelerator_enumerate_mmios(enyx_hw_accelerator * accelerator,
        enyx_hw_filter const * filter,
        enyx_hw_mmio_descriptor const ** descriptors, size_t capacity);

/**
 * Free descriptors allocated by @ref enyx_hw_accelerator_enumerate_all_mmios
 * or @ref enyx_hw_accelerator_enumerate_mmios
 *
 * @param descriptors Array to be freed
 * @param capacity Size of descriptors.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_free_mmio_descriptors(enyx_hw_mmio_descriptor const ** descriptors,
        size_t capacity);

/**
 * Get the number of c2a_streams for this accelerator.
 * @param accelerator Accelerator to count c2a_streams for
 * @return The number of c2a_streams on success, 0 with errno set on error.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY size_t
enyx_hw_accelerator_count_c2a_streams(enyx_hw_accelerator * accelerator);

/**
 * Enumerate all c2a_stream descriptors for a given accelerator.
 *
 * @param accelerator Accelerator used.
 * @param [out] descriptors Array of @ref enyx_hw_c2a_stream_descriptor.
 * @param capacity size of descriptors.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_accelerator_enumerate_all_c2a_streams(enyx_hw_accelerator * accelerator,
        enyx_hw_c2a_stream_descriptor const ** descriptors, size_t capacity);

/**
 * Enumerate c2a_stream descriptors matching filter for a given accelerator.
 *
 * @param accelerator Accelerator used.
 * @param filter filter to check
 * @param [out] descriptors Array of @ref enyx_hw_c2a_stream_descriptor.
 * @param capacity size of descriptors.
 * @return The count of selected c2a_streams on success, -1 with errno set on
 *         failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_accelerator_enumerate_c2a_streams(enyx_hw_accelerator * accelerator,
        enyx_hw_filter const * filter,
        enyx_hw_c2a_stream_descriptor const ** descriptors, size_t capacity);

/**
 * Free descriptors allocated by @ref enyx_hw_accelerator_enumerate_all_c2a_streams
 * or @ref enyx_hw_accelerator_enumerate_c2a_streams
 *
 * @param descriptors Array to be freed
 * @param capacity Size of descriptors.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_free_c2a_stream_descriptors(enyx_hw_c2a_stream_descriptor const ** descriptors,
        size_t capacity);

/**
 * Get the number of a2c_streams for this accelerator.
 * @param accelerator Accelerator to count a2c_streams for
 * @return The number of a2c_streams on success, 0 with errno set on error.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY size_t
enyx_hw_accelerator_count_a2c_streams(enyx_hw_accelerator * accelerator);

/**
 * Enumerate all a2c_stream descriptors for a given accelerator.
 *
 * @param accelerator Accelerator used.
 * @param [out] descriptors Array of @ref enyx_hw_a2c_stream_descriptor.
 * @param capacity size of descriptors.
 * @return 0 on success, -1 with errno set on failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_accelerator_enumerate_all_a2c_streams(enyx_hw_accelerator * accelerator,
        enyx_hw_a2c_stream_descriptor const ** descriptors, size_t capacity);

/**
 * Enumerate a2c_stream descriptors matching filter for a given accelerator.
 *
 * @param accelerator Accelerator used.
 * @param filter filter to check
 * @param [out] descriptors Array of @ref enyx_hw_a2c_stream_descriptor.
 * @param capacity size of descriptors.
 * @return The count of selected a2c_streams on success, -1 with errno set on
 *         failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hw_accelerator_enumerate_a2c_streams(enyx_hw_accelerator * accelerator,
        enyx_hw_filter const * filter,
        enyx_hw_a2c_stream_descriptor const ** descriptors, size_t capacity);

/**
 * Free descriptors allocated by @ref enyx_hw_accelerator_enumerate_all_a2c_streams
 * or @ref enyx_hw_accelerator_enumerate_a2c_streams
 *
 * @param descriptors Array to be freed
 * @param capacity Size of descriptors.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_free_a2c_stream_descriptors(enyx_hw_a2c_stream_descriptor const ** descriptors,
        size_t capacity);
