/**
 *
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator to CPU data streaming.
 */
#pragma once

#include <stdint.h>
#include <stdbool.h>

#include <enyx/hw_c/properties.h>
#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/fwd.h>

/**
 * This class references a a2c_stream
 *
 * It is used to construct a a2c_stream, but doesn't
 * constitute ownership of it.
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_a2c_stream_descriptor enyx_hw_a2c_stream_descriptor;

/**
 * Retrieve the properties associated with a a2c stream @p descriptor
 *
 * @param descriptor The a2c stream descriptor object to query
 * @returns A pointer to the properties object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_properties const *
enyx_hw_a2c_stream_descriptor_get_properties(enyx_hw_a2c_stream_descriptor const * descriptor);

/**
 * Retrieve the stream mtu
 *
 * @param descriptor the a2c stream descriptor to query
 * @param mtu the integer to fill with the mtu
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_mtu(enyx_hw_a2c_stream_descriptor const * descriptor,
                                      uint32_t * mtu);

/**
 * Retrieve the stream size
 *
 * @param descriptor the a2c stream descriptor to query
 * @param size the integer to fill with the size
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_size(enyx_hw_a2c_stream_descriptor const * descriptor,
                                       size_t * size);

/**
 * Retrieve the packet_count statistic.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 * @note This does not account for packets that were dropped due to errors.
 *
 * @param descriptor the descriptor to query
 * @param[out] packet_count The output packet_count.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_packet_count(enyx_hw_a2c_stream_descriptor const * descriptor,
                                               uint32_t * packet_count);

/**
 * Retrieve the current number of packets that were back-pressured in the
 * accelerator due to lack of space in the accelerator to CPU buffer.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param descriptor the descriptor to query
 * @param[out] backpressure_count The output backpressure_count.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_backpressure(enyx_hw_a2c_stream_descriptor const * descriptor,
                                                     uint32_t * backpressure_count);
/**
 * Retrieve the current number of packets dropped by the accelerator because
 * the accelerator to CPU buffer was full.
 *
 * @note Packets are dropped by the accelerator *only* if drop mode is enabled
 * for this device.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param descriptor the descriptor to query
 * @param[out] fifo_errors The output fifo_errors.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_fifo_errors(enyx_hw_a2c_stream_descriptor const * descriptor,
                                              uint32_t * fifo_errors);

/**
 * Retrieve the current number of packets dropped by the accelerator because
 * the packet was bigger than the device MTU.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param descriptor the descriptor to query
 * @param[out] truncated_count The output truncated_count.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_truncated_count(enyx_hw_a2c_stream_descriptor const * descriptor,
                                                  uint32_t * truncated_count);

/**
 * Retrieve the current number of ingress packets with the error bit set to 1.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param descriptor the descriptor to query
 * @param[out] errors The output errors.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_descriptor_get_errors(enyx_hw_a2c_stream_descriptor const * descriptor,
                                         uint32_t * errors);

/**
 * Accelerator to CPU stream configuration
 *
 * @since 5.0.0
 */
typedef struct
{
    /// Enable monitoring of usage. See @ref enyx_hw_a2c_stream_stats_get_usage
    bool enable_usage;

    /// Get a a2c_stream of size 2^buffer_width (0 means default size).
    uint32_t buffer_width;
} enyx_hw_a2c_stream_configuration;

/**
 * Represents the handler invoked by the enyx_hw_a2c_stream_poll()
 *
 * @since 5.0.0
 */
typedef void
(*enyx_hw_a2c_stream_handler)(uint8_t const * data, uint32_t size, void * opaque);

/**
 * Create an Accelerator to CPU channel from an @p descriptor and a @p c
 *
 * @param descriptor The stream descriptor
 * @param c The stream configuration. Should be NULL if no specific
 *          configuration is used.
 * @returns The created stream object or NULL if an error
 *          occurred (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_a2c_stream *
enyx_hw_a2c_stream_create(enyx_hw_a2c_stream_descriptor const* descriptor,
                          enyx_hw_a2c_stream_configuration const* c);

/**
 * Detroy a stream object @p c
 *
 * @param s The stream to destroy
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_a2c_stream_destroy(enyx_hw_a2c_stream * s);

/**
 * Retrieve the accelerator associated with a a2c_stream descriptor @p d
 *
 * @param d The descriptor to query
 * @returns A pointer to the accelerator object
 *
 * @since 5.5.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_accelerator *
enyx_hw_a2c_stream_get_accelerator(enyx_hw_a2c_stream_descriptor const * d);

/**
 * Retrieve the descriptor associated with a a2c stream @p c
 *
 * @param s The a2c stream object to query
 * @returns A pointer to the descriptor object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_a2c_stream_descriptor const *
enyx_hw_a2c_stream_get_descriptor(enyx_hw_a2c_stream const * s);

/**
 * Free a previously getted descriptor.
 *
 * @param descriptor The descriptor to free.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_a2c_stream_put_descriptor(enyx_hw_a2c_stream_descriptor const * descriptor);

/**
 * Retrieve the stream mtu
 *
 * @param s The a2c stream object to query
 * @param mtu The integer to fill with the mtu
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.6.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_get_mtu(enyx_hw_a2c_stream const * s, uint32_t * mtu);

/**
 * Retrieve the stream size
 *
 * @param s The a2c stream object to query
 * @param size The integer to fill with the size
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_get_size(enyx_hw_a2c_stream const * s, size_t * size);

/**
 * Poll to receive data and invokes the @p handler provided to the
 * constructor.
 *
 * @param s The a2c stream to use
 * @param on_data The handler called on each packet
 * @param opaque Opaque pointer passed to the @a on_data callback
 * @returns @b 1 if a message has been received
 *          (i.e. @p handler invoked); otherwise @b 0.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY size_t
enyx_hw_a2c_stream_poll_once(enyx_hw_a2c_stream const * s,
                             enyx_hw_a2c_stream_handler on_data,
                             void * opaque);

/**
 * Retrieve the packet_count statistic.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 * @note This does not account for packets that were dropped due to errors.
 *
 * @param s The stats object to query
 * @param[out] packet_count The output packet_count.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_stats_get_packet_count(enyx_hw_a2c_stream const * s,
                                               uint32_t * packet_count);

/**
 * Retrieve the current number of packets that were back-pressured in the
 * accelerator due to lack of space in the accelerator to CPU buffer.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param s The stats object to query
 * @param[out] backpressure The output backpressure.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_stats_get_backpressure(enyx_hw_a2c_stream const * s,
                                               uint32_t * backpressure);

/**
 * Retrieve the current number of packets dropped by the accelerator because
 * the accelerator to CPU buffer was full.
 *
 * @note Packets are dropped by the accelerator *only* if drop mode is enabled
 * for this device.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param s The stats object to query
 * @param[out] fifo_errors The output fifo_errors.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_stats_get_fifo_errors(enyx_hw_a2c_stream const * s,
                                              uint32_t * fifo_errors);

/**
 * Retrieve the current number of packets dropped by the accelerator because
 * the packet was bigger than the device MTU.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param s The stats object to query
 * @param[out] truncated_count The output truncated count.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.12.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_stats_get_truncated_count(enyx_hw_a2c_stream const * s,
                                             uint32_t * truncated_count);

/**
 * Retrieve the current number of ingress packets with the error bit set to 1.
 *
 * @warning Getting this statistic is a slow operation as it needs to do
 * syscalls and MM read/write operations. It should not be done in an
 * application's critical path.
 *
 * @param s The stats object to query
 * @param[out] errors The output errors.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_stats_get_errors(enyx_hw_a2c_stream const * s,
                                         uint32_t * errors);

/**
 * Retrieve the current free space in the buffer.
 *
 * @note This statistic is only available if *enable_usage* was set when
 * opening this device.
 *
 * @param s The stats object to query
 * @param[out] usage The current buffer usage.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_a2c_stream_stats_get_usage(enyx_hw_a2c_stream const * s,
                                        uint32_t * usage);
