ENYX_HW_NAMESPACE_BEGIN

namespace mocking {

inline
core::core(std::uint8_t major,
           std::uint8_t minor,
           std::uint16_t revision,
           std::uint16_t hardware_id,
           std::uint8_t addr_width,
           std::uint64_t base_addr) noexcept
    : core({major, minor, revision, hardware_id}, addr_width, base_addr)
{}

inline
core::core(const hw::core::descriptor_type & descriptor,
           std::uint8_t addr_width,
           std::uint64_t base_addr) noexcept
    : core_(::enyx_hwm_core_create(&descriptor,
                                   addr_width,
                                   base_addr),
            &::enyx_hwm_core_destroy)
    , registers_()
    , children_()
{
    auto reset_cb = [] (void * context) -> int {
        auto thiz = reinterpret_cast<core *>(context);
        if (! thiz->on_reset) {
            errno = EINVAL;
            return -1;
        }
        auto res = thiz->on_reset();
        if (! res) {
            errno = res.e().value();
            return -1;
        }
        return 0;
    };
    ::enyx_hwm_core_set_reset_callback(handle(), reset_cb, this);
}

inline
mocked_register &
core::add_register(register_description const& desc)
{
    registers_.emplace_back(handle(), desc);
    return registers_.back();
}

inline
mocked_register &
core::add_register(std::uint64_t byte_offset,
                          std::uint8_t bit_offset,
                          std::uint8_t bit_width)
{
    return add_register({byte_offset, bit_offset, bit_width});
}

template<typename MockedCore, typename... Args>
inline
MockedCore &
core::add_child(Args&&... args)
{
    std::unique_ptr<MockedCore> ptr{new MockedCore{args...}};
    MockedCore & ret = *ptr;
    if (::enyx_hwm_core_bind_child(handle(), ret.handle()) < 0)
        throw std::system_error{errno, std::generic_category(),
                                "Could not add child to core"};
    children_.emplace_back(std::move(ptr));
    return ret;
}

inline
std::vector<core::core_ref>
core::get_children() noexcept
{
    std::vector<core::core_ref> ret;
    for (auto & child: children_)
        ret.emplace_back(*child);
    return ret;
}

inline
std::vector<core::register_ref>
core::get_registers() noexcept
{
    std::vector<core::register_ref> ret;
    for (auto & entry: registers_)
        ret.emplace_back(std::ref(entry));
    return ret;
}

namespace {
template<typename UINTX_t>
struct _core_uintx_t_ops;

template<>
struct _core_uintx_t_ops<std::uint8_t> {
    static int
    write(::enyx_hwm_core * core, uint64_t address, uint8_t value) {
        return ::enyx_hwm_core_write_8(core, address, value);
    }

    static int
    read(::enyx_hwm_core * core, uint64_t address, uint8_t * value) {
        return ::enyx_hwm_core_read_8(core, address, value);
    }
};


template<>
struct _core_uintx_t_ops<std::uint16_t> {
    static int
    write(::enyx_hwm_core * core, uint64_t address, uint16_t value) {
        return ::enyx_hwm_core_write_16(core, address, value);
    }

    static int
    read(::enyx_hwm_core * core, uint64_t address, uint16_t * value) {
        return ::enyx_hwm_core_read_16(core, address, value);
    }
};

template<>
struct _core_uintx_t_ops<std::uint32_t> {
    static int
    write(::enyx_hwm_core * core, uint64_t address, uint32_t value) {
        return ::enyx_hwm_core_write_32(core, address, value);
    }

    static int
    read(::enyx_hwm_core * core, uint64_t address, uint32_t * value) {
        return ::enyx_hwm_core_read_32(core, address, value);
    }
};

template<>
struct _core_uintx_t_ops<std::uint64_t> {
    static int
    write(::enyx_hwm_core * core, uint64_t address, uint64_t value) {
        return ::enyx_hwm_core_write_64(core, address, value);
    }

    static int
    read(::enyx_hwm_core * core, uint64_t address, uint64_t * value) {
        return ::enyx_hwm_core_read_64(core, address, value);
    }
};

}

template<typename UINTX_T>
inline
result<void>
core::write(uint64_t address, UINTX_T value) noexcept
{
    if (_core_uintx_t_ops<UINTX_T>::write(handle(), address, value) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

template<typename UINTX_T>
inline
result<UINTX_T>
core::read(uint64_t address) noexcept
{
    UINTX_T value = 0;
    if (_core_uintx_t_ops<UINTX_T>::read(handle(), address, &value) != 0)
        return std::error_code{errno, std::generic_category()};
    return value;
}

inline
hw::core::descriptor_type const&
core::get_descriptor(void) const noexcept
{
    return *::enyx_hwm_core_get_descriptor(handle());
}

inline
result<void>
core::set_product_version(hw::product_version const & version) noexcept
{
    if (::enyx_hwm_core_set_product_version(handle(), &version) < 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline
void
core::update_addresses(std::uint8_t new_addr_width,
                       std::uint64_t new_base_addr)
{
    if (::enyx_hwm_core_update_addresses(handle(), new_addr_width,
                                                new_base_addr) != 0) {
        throw std::system_error{errno, std::generic_category(),
                                "Failed to update addresses"};
    }
}

inline
uint8_t
core::get_required_width(void) const noexcept
{
    return ::enyx_hwm_core_get_required_width(handle());
}

inline
std::uint64_t
core::get_base_address(void) const noexcept
{
    return ::enyx_hwm_core_get_base_address(handle());
}

inline
core_address_space const&
core::get_address_space(void) const noexcept
{
    return *::enyx_hwm_core_get_address_space(handle());
}


inline ::enyx_hwm_core const *
core::handle() const noexcept {
    return core_.get();
}

inline ::enyx_hwm_core *
core::handle() noexcept {
    return core_.get();
}

} /* namespace mocking */

ENYX_HW_NAMESPACE_END
