#pragma once

#include <cstdint>
#include <functional>
#include <list>
#include <memory>
#include <vector>

#include <enyx/hw/core.hpp>
#include <enyx/hw/mocking/register.hpp>
#include <enyx/hw/register_description.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/hw_c/mocking/core.h>

///@cond
namespace std {
template<>
struct default_delete<::enyx_hwm_core>
{
    void
    operator()(::enyx_hwm_core * ptr) const
    {
        ::enyx_hwm_core_destroy(ptr);
    }
};
}
///@endcond

ENYX_HW_NAMESPACE_BEGIN

namespace mocking {

/**
 *  This class is used to represent a mocked core
 *
 *  Registers and mocked core children can be added
 *  to mimic to real hardware.
 *
 *  @note This class can be overrided
 */
class core
{
public:
    /**
     * C core pointer type
     */
    using core_ptr = std::shared_ptr<::enyx_hwm_core>;
    /**
     * Core reference type
     */
    using core_ref = std::reference_wrapper<core>;

    /**
     * Register reference type
     */
    using register_ref = std::reference_wrapper<mocked_register>;

public:
    /**
     * Construct a core from its metadata
     *
     * @param major core major version
     * @param minor core minor version
     * @param revision core revision
     * @param hardware_id core hardware_id
     * @param addr_width The core address width
     * @param base_addr The core base address
     */
    core(std::uint8_t major,
         std::uint8_t minor,
         std::uint16_t revision,
         std::uint16_t hardware_id,
         std::uint8_t addr_width = 32,
         std::uint64_t base_addr = 0) noexcept;

    /**
     * Construct a core from an existing descriptor
     *
     * @param descriptor The descriptor to be used
     * @param addr_width The core address width
     * @param base_addr The core base address
     */
    core(const hw::core::descriptor_type & descriptor,
         std::uint8_t addr_width = 32,
         std::uint64_t base_addr = 0) noexcept;

    /**
     * Destructor
     */
    virtual ~core() = default;

    /**
     * Deleted copy constructor.
     */
    core(core const &) = delete;

    /**
     * Deleted copy assignment operator.
     */
    core operator=(core const &) = delete;

    /**
     * Add a register to this core.
     *
     * @param description The register description.
     *
     * @return A reference to the new register.
     * @throws std::system_error if addr_width is already set or if it is too small.
     */
    mocked_register &
    add_register(register_description const& description);

    /**
     * Add a register to this core.
     *
     * @param byte_offset The register byte-offset from the core base address.
     * @param bit_offset The register bit-offset from the byte-offset.
     * @param bit_width The register size in bits.
     *
     * @return A reference to the new register.
     * @throws std::system_error if addr_width is already set or if it is too small.
     */
    mocked_register &
    add_register(std::uint64_t byte_offset,
                 std::uint8_t bit_offset,
                 std::uint8_t bit_width);

    /**
     * Add a new child core to this core.
     *
     * @tparam MockedCore Type of the new core
     * @tparam Args Type of MockedCore's constructor arguments.
     * @param args Arguments of MockedCore's constructor
     * @return A pointer to the new core.
     * @throws std::system_error if addr_width is already set or if it is too small.
     */
    template<typename MockedCore = core, typename... Args>
    MockedCore &
    add_child(Args&&... args);

    /**
     * Get a vector of references to children.
     *
     * @return A vector containing references to this core's children.
     */
    std::vector<core_ref>
    get_children() noexcept;

    /**
     * Get a vector of references to registers.
     *
     * @return A vector containing references to this core's registers.
     */
    std::vector<register_ref>
    get_registers() noexcept;

    /**
     * Write a value to local registers matching address.
     *
     * @tparam UINTX_T type of value to write (uin8_t, uint16_t, uint32_t
     *                                         uint64_t).
     * @param address Absolute address for the write.
     * @param value Value to write
     *
     * @return a result object containing an error on failure.
     */
    template<typename UINTX_T>
    result<void>
    write(uint64_t address, UINTX_T value) noexcept;

    /**
     * Read a value from local registers matching address.
     *
     * @tparam UINTX_T type of value to read (uin8_t, uint16_t, uint32_t
     *                                        uint64_t).
     * @param address Absolute address for the read.
     *
     * @return a result object containing the value on succes or an error on
     * failure.
     */
    template<typename UINTX_T>
    result<UINTX_T>
    read(uint64_t address) noexcept;

    /**
     * Retrieve the core descriptor
     *
     * @return A reference to the descriptor
     */
    hw::core::descriptor_type const&
    get_descriptor(void) const noexcept;

    /**
     * Set the product version of this core.
     *
     * @param version The version to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_product_version(hw::product_version const & version) noexcept;

    /**
     *  Update the address space width and base address of the core and its
     *  subtree.
     *
     *  @param new_addr_width The new address width
     *  @param new_base_addr The new base_address
     *
     *  @note The address width and base address of the children will be
     *        updated as well.
     */
    void
    update_addresses(std::uint8_t new_addr_width,
                     std::uint64_t new_base_addr);

    /**
     *  Get the required width of the current core
     *
     *  The required width is deduced from the count of
     *  children.
     *
     *  @return The width as a power of 2.
     */
    std::uint8_t
    get_required_width(void) const noexcept;

    /**
     *  Get the base address of the current core in its current hierarchy
     *
     *  This base address is calculated by traversing the core tree from the
     *  root node and using core address widths.
     *
     *  @return The base address of the core.
     */
    std::uint64_t
    get_base_address(void) const noexcept;

    /**
     *  Get a reference to core address space as a sequence of bytes
     *
     *  @return A const reference to a sequence of bytes
     */
    core_address_space const&
    get_address_space(void) const noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_core const *
    handle() const noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_core *
    handle() noexcept;

public:
    /// Signature of the `on_reset` callback
    using on_reset_sig = result<void>();

    /**
     * Callback called on core reset.
     */
    std::function<on_reset_sig> on_reset;

private:
    std::shared_ptr<::enyx_hwm_core> core_;
    std::list<mocked_register> registers_;
    std::vector<std::unique_ptr<core>> children_;
};

/// @cond
using mocked_core
    ENYX_HW_CXX_DEPRECATED("Replaced by mocking::core") = core;
/// @endcond

} /* namespace mocking */

ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/core.ipp>
