#include "Color.hpp"

#include <cstdio>
#include <unistd.h>
#include <ostream>
#include <sstream>

#include <map>

namespace {

bool _enabled = true;

}

void
enable_color(ColorConfiguration const & configuration)
{
    switch (configuration) {
    case ColorConfiguration::NEVER:
        _enabled = false;
        return;
    case ColorConfiguration::AUTO:
        if (::isatty(::fileno(stdout)) != 1) {
            _enabled = false;
            return;
        }
    /* Fallthrough */
    case ColorConfiguration::ALWAYS:
        _enabled = true;
    }
}

std::istream &
operator>>(std::istream & in, ColorConfiguration & color_configuration)
{
    std::string str;
    in >> str;

    if (str == "never")
        color_configuration = ColorConfiguration::NEVER;
    else if (str == "auto")
        color_configuration = ColorConfiguration::AUTO;
    else if (str == "always")
        color_configuration = ColorConfiguration::ALWAYS;
    else {
        std::ostringstream error;
        error << "Unknown color setting '" << str << "'";
        throw std::runtime_error{error.str()};
    }
    return in;
}

#define _RGB(r, g, b) "\e[38;2;" #r ";" #g ";" #b "m"

constexpr auto CB_BLACK = _RGB(0, 0, 0);
constexpr auto CB_ORANGE = _RGB(230, 159, 0);
constexpr auto CB_SKY_BLUE = _RGB(86, 180, 233);
constexpr auto CB_BLUEISH_GREEN = _RGB(0, 158, 115);
constexpr auto CB_YELLOW = _RGB(240, 228, 66);
constexpr auto CB_BLUE = _RGB(0, 114, 178);
constexpr auto CB_VERMILLION=  _RGB(213, 94, 0);
constexpr auto CB_REDDISH_PURPLE = _RGB(204, 121, 167);

constexpr auto RED = "\e[31m";
constexpr auto GREEN = "\e[32m";
constexpr auto YELLOW = "\e[33m";
constexpr auto BLUE = "\e[34m";
constexpr auto MAGENTA = "\e[35m";
constexpr auto CYAN = "\e[36m";
constexpr auto WHITE = "\e[37m";

constexpr auto BOLD_RED = "\e[1;31m";
constexpr auto BOLD_GREEN = "\e[1;32m";
constexpr auto BOLD_YELLOW = "\e[1;33m";
constexpr auto BOLD_BLUE = "\e[1;34m";
constexpr auto BOLD_MAGENTA = "\e[1;35m";
constexpr auto BOLD_CYAN = "\e[1;36m";
constexpr auto BOLD_WHITE = "\e[1;37m";

constexpr auto BOLD = "\e[1m";

constexpr auto CLEAR = "\e[0m";

static const std::map<Color, char const *> _colors{
    {Color::IFNAME, CB_SKY_BLUE},
    {Color::MAC, CB_YELLOW},
    {Color::INET, CB_REDDISH_PURPLE},
    {Color::CONFIGURED, CB_BLUEISH_GREEN},
    {Color::UNCONFIGURED, CB_VERMILLION},
    {Color::BOLD, BOLD},
    {Color::NONE, CLEAR},
};

std::ostream &
operator<<(std::ostream & out, Color const & color)
{
    if (_enabled)
        out << _colors.at(color);
    return out;
}
