#pragma once

#include <stdint.h>
#include <stdbool.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/data_stream/source.h>

/**
 *  A structure representing the EMI header.
 */
typedef struct
{
    /// The event type as described by @ref enyx_tcp_event_type
    uint32_t type;
} enyx_tcp_event_header;

/**
 *  Represents the type of the EMI event.
 */
enum enyx_tcp_event_type
{
    // The status event identifier
    ENYX_TCP_EVENT_STATUS = 1,
    /// The credit event identifier
    ENYX_TCP_EVENT_CREDIT = 2,
};

/**
 *  Represents a credit event
 */
typedef struct
{
    // The latest absolute credit value
    uint32_t credit;
} enyx_tcp_credit_event;

/**
 *  Represent all the software visible TCP states.
 */
typedef enum {
	/**
	 * The TCP session is closed. TCP state:
	 * * CLOSED
	 */
	ENYX_TCP_STATE_CLOSED = 0,

	/**
	 * The TCP session is being opened. TCP states:
	 * * LISTEN
	 * * SYN-SENT
	 * * SYN-RCVD
	 */
	ENYX_TCP_STATE_OPENING = 1,

	/**
	 * The TCP session is established, transfers can occur. TCP state:
	 * * ESTABLISHED
	 */
	ENYX_TCP_STATE_ESTABLISHED = 2,

	/**
	 * The TCP session is being closed. TCP states:
	 * * FIN-WAIT1
	 * * FIN-WAIT2
	 * * CLOSE_WAIT
	 * * CLOSING
	 * * LAST-ACK
	 * * TIME-WAIT
	 */
	ENYX_TCP_STATE_CLOSING = 3,
} enyx_tcp_state;

/**
 *  Represents a status event
 */
typedef struct
{
    /// The new session status as described by @ref enyx_tcp_state
    uint32_t session_status;
    /// The remote peer ip when state is ENYX_TCP_STATE_ESTABLISHED
    uint32_t peer_ip;
    /// The remote peer port when state is ENYX_TCP_STATE_ESTABLISHED
    uint16_t peer_port;
    uint8_t pad[2];
} enyx_tcp_session_status_event;

/**
 *  This is the class used to retrieve the TCP session
 *  status and credits from a @b TOE.
 *
 *  Using the credits and status, the user application
 *  may delay or abort the sending of data to specific session.
 *
 *  Usage:
 *  @code
 *  int result = -1;
 *
 *  // Create the tcp context
 *  enyx_tcp_context * context;
 *  context = enyx_tcp_context_create(on_state_change, on_credit_change
 *                                           on_error);
 *  if (! context)
 *  {
 *      perror("enyx_tcp_context_create");
 *      goto create_failed;
 *  }
 *
 *  // Create the TCP context source
 *  enyx_data_stream_source tcp_context_source = {
 *      .opaque = context,
 *      .on_data = enyx_tcp_context_on_emi_data,
 *  };
 *
 *  // Poll for events until is_exit_requested == true
 *  do
 *      result = enyx_data_stream_hw_source_poll_once(a2c_stream, tcp_context_source);
 *  while (! is_exit_requested && result >= 0);
 *  if (result < 0)
 *      perror("enyx_data_stream_hw_source_poll_once");
 *
 *  enyx_tcp_context_destroy(context);
 *  create_failed:
 *  return result;
 *  @endcode
 *
 */
typedef struct enyx_tcp_context enyx_tcp_context;

/**
 *  Union of all the state info
 *
 *  The actual type can be either @ref enyx_tcp_state_info::established_t or
 *  @ref enyx_tcp_state_info::closed_t.
 */
typedef union {
    /// Contain the established state info
    struct established_t {
        /// The remote peer ip
        uint32_t peer_ip;

        /// The remote peer port
        uint16_t peer_port;
    } established;

    /// Contain the closed state info
    struct closed_t {
        /// Is the closed caused by a peer reset
        bool peer_reset;
    } closed;
} enyx_tcp_state_info;

/**
 *  This callback is called on each TCP session state change.
 *
 *  @note According to new_state, the @p state_info @b union
 *        may contain different data types:
 *        - @b established member is populated
 *          when @p new_state == ENYX_TCP_STATE_ESTABLISHED
 *        - @b closed member is populated
 *          when @p new_state == ENYX_TCP_STATE_CLOSED
 */
typedef void
(*enyx_tcp_on_state_change)(uint32_t session_id,
                            enyx_tcp_state new_state,
                            enyx_tcp_state_info const* state_info,
                            void * opaque);

/**
 *  This callback is called when TCP session @p session_id
 *  @p credit changes significantly.
 *
 *  @note The @p credit value is an absolute value of the TOE
 *        current credit, as opposed to a relative value.
 */
typedef void
(*enyx_tcp_on_credit_change)(uint32_t session_id,
	                         uint32_t credit,
                             void * opaque);

/**
 *  This callback is called on TCP emi data error with @b errno set accordingly.
 */
typedef void
(*enyx_tcp_on_emi_error)(void * opaque);

/**
 *  Create an io_context invoking @p on_state_change &
 *  @p on_credit_change accordingly, and @p on_emi_error on error.
 *
 *  @param on_state_change The callback to invoke on state change
 *  @param on_credit_change The callback to invoke on credit change
 *  @param on_emi_error The callback to invoke on error
 *  @param opaque An opaque pointer to forward to the @p handlers.
 *  @return The created stream object or NULL if an error
 *          occurred (@b errno is set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_tcp_context *
enyx_tcp_context_create(enyx_tcp_on_state_change on_state_change,
                        enyx_tcp_on_credit_change on_credit_change,
                        enyx_tcp_on_emi_error on_emi_error,
                        void * opaque);

/**
 * TCP context callback to invoke when EMI data is available.
 *
 * @param data: The EMI data to decode
 * @param size: Data size
 * @param metadata: Metadata associated with passed data
 * @param opaque: Opaque pointer to TCP context
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_tcp_context_on_emi_data(uint8_t const * data,
                             uint32_t size,
                             void const  * metadata,
                             void * opaque);

/**
 * Detroy a context object @p c
 *
 * @param c The context to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_tcp_context_destroy(enyx_tcp_context * c);
