#pragma once

#include <stddef.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>

#include <enyx/cores_c/probes/event.h>
#include <enyx/cores_c/probes/probe.h>

/**
 * Represents an event(s) source
 *
 * Created from the @ref enyx_probes_collector, the event
 * source invokes a user-provided handler for each
 * @ref enyx_probes_probe_event generated by the hardware.
 *
 * Using the probe event's reference to the associated probe,
 * the width of the event's fields (i.e. reference_id, custom_info) can
 * be retrieved using @ref enyx_probes_probe_get_event_format.
 *
 * @since 6.7.0
 */
typedef struct enyx_probes_event_source enyx_probes_event_source;

/**
 * Represents the handler invoked by the enyx_probes_event_source_poll()
 *
 * @param probe_id The ID of probe which triggered the event.
 * @param event The triggered event.
 * @param opaque The data pointer passed at event source creation.
 *
 * @since 6.7.0
 */
typedef void
(*enyx_probes_event_handler)(enyx_probes_probe_id probe_id,
                             enyx_probes_probe_event const* event,
                             void * opaque);

/**
 * Represents the handler invoked on error.
 *
 * @param opaque The data pointer passed at event source creation.
 *
 * @since 6.7.0
 */
typedef void
(*enyx_probes_event_error)(void * opaque);


/**
 * Create a probe event source to retrieve probe events from a collector DMA
 * channel.
 *
 * @param on_event The callback to invoke on probe event.
 * @param on_error The callback to invoke on invalid probe event.
 * @param opaque A pointer to context data to pass to callbacks.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_event_source *
enyx_probes_event_source_create(enyx_probes_event_handler on_event,
                                 enyx_probes_event_error on_error,
                                 void * opaque);

/**
 * Destroy a probe event source.
 *
 * @param source The source to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_probes_event_source_destroy(enyx_probes_event_source * source);

/**
 * Probes source callback to invoke when data to be decoded is available.
 *
 * @param data: The probes data to decode
 * @param size: Data size
 * @param metadata: Metadata associated with passed data
 * @param opaque: Opaque pointer to probes source
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_probes_on_data(uint8_t const * data,
                    uint32_t size,
                    void const  * metadata,
                    void * opaque);
