#pragma once

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>

/**
 * The main memory tester object.
 */
typedef struct enyx_memory_tester enyx_memory_tester;

/**
 * Parameters to set for the next run.
 */
typedef struct {
    /// First test to run. Maximum value is 15.
    uint8_t first_test : 4;
    /// Last test to run. Maximum value is 15.
    uint8_t last_test : 4;
    /// Number of test sequences to run.
    uint8_t loops;
    /**
     * Enable random access.
     *
     * When enabled, random delay between memory requests will be introduced.
     */
    bool random_access;
    /**
     * Enable random byte access.
     *
     * When enabled, randomly chosen bytes will not be marked as valid in write
     * requests.
     */
    bool random_byte_access;
} enyx_memory_tester_parameters;

/**
 * Enumerate and create memory tester objects for all ``enyx_memory_tester_v2``
 * HW cores found.
 *
 * @param root The root core
 * @param out An array of memory tester objects to fill
 * @param size The size of the @b out array
 * @return The number of ``enyx_memory_tester_v2`` HW cores found on success,
 * -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_memory_tester_enumerate(enyx_hw_core const * root,
                             enyx_memory_tester ** out, size_t size);

/**
 * Create the memory tester object from a ``enyx_memory_tester_v2`` HW core.
 *
 * @param core The ``enyx_memory_tester_v2`` core to use
 * @return An @ref enyx_memory_tester object on success, @b NULL on failure
 * (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_memory_tester *
enyx_memory_tester_create(enyx_hw_core * core);

/**
 * Destroy a memory tester object.
 *
 * @param memtest The memory tester object to use
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_memory_tester_destroy(enyx_memory_tester * memtest);

/**
 * Set memory test parameters.
 *
 * @param memtest The memory tester object to use
 * @param params The memory tester parameters to set
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_set_parameters(enyx_memory_tester * memtest,
                                  enyx_memory_tester_parameters const * params);

/**
 * Start a memory test run.
 *
 * To configure the test run, use the @ref enyx_memory_tester_set_parameters
 * function with the desired parameters.
 *
 * @param memtest The memory tester object to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_start(enyx_memory_tester * memtest);

/**
 * Stop a running memory test.
 *
 * @param memtest The memory tester object to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_stop(enyx_memory_tester * memtest);

/**
 * Clear current errors.
 *
 * @param memtest The memory tester object to use
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_clear_error(enyx_memory_tester * memtest);

/**
 * Check if a memory test have finished.
 *
 * @param memtest The memory tester object to use
 * @param done The memory test status
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_is_done(enyx_memory_tester const * memtest, bool * done);

/**
 * Check if a memory test has failed.
 *
 * @param memtest The memory tester object to use
 * @param failed The memory test failure status
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_has_failed(enyx_memory_tester const * memtest,
                              bool * failed);

/**
 * Get the last failed test number.
 *
 * This value is only valid when @ref enyx_memory_tester_has_failed is @b true.
 *
 * @param memtest The memory tester object to use
 * @param fail The last failed test number
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_memory_tester_get_last_fail(enyx_memory_tester const * memtest,
                                 uint8_t *fail);

/**
 * Get the number of failed memory tests.
 *
 * @param memtest The memory tester object to use
 * @return The number of failed test on success, -1 on failure (with @b errno
 * set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_memory_tester_get_fail_count(enyx_memory_tester const * memtest);
