#pragma once

#include <stddef.h>
#include <stdint.h>
#include <time.h>
#include <unistd.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/types.h>
#include <enyx/hw_c/core.h>

/**
 * @brief HW top management object.
 */
typedef struct enyx_hw_top enyx_hw_top;

/**
 * FPGA Serial ID type. Bytes are in little endian.
 *
 * For example with the serial ID: 112233445566778899aabbcc
 * @code
 * enyx_hw_top_fpga_serial_id = {
 *     .value = { 0xcc, 0xbb, 0xaa, 0x99,
 *                0x88, 0x77, 0x66, 0x55,
 *                0x44, 0x33, 0x22, 0x11 }
 * };
 * @endcode
 */
typedef struct {
    uint8_t value[12];
} enyx_hw_top_fpga_serial_id;

/// Maximum length of hw_top core string register plus NUL character
ENYX_CORES_C_SYMBOL_VISIBILITY
uint16_t ENYX_HW_TOP_STRLEN_MAX;

/**
 * Print a @p serial id into a @p buffer.
 *
 * @param serial The serial ID to print
 * @param[out] buffer The buffer to print into
 * @param buffer_capacity The capacity of @p buffer
 *
 * @return The number of character of the string representation on success,
 *         -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_hw_top_print_fpga_serial_id(enyx_hw_top_fpga_serial_id const * serial,
                                 char * buffer,
                                 size_t buffer_capacity);

/**
 * @brief Get the top build date as a UNIX timestamp.
 *
 * @param hw_top The HW top object.
 * @param timestamp The build date timestamp.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_build_date(const enyx_hw_top * hw_top, time_t * timestamp);

/**
 * @brief Get the random seed value set by the build scripts.
 *
 * @param hw_top The HW top object.
 * @param seed The seed value
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_seed(const enyx_hw_top * hw_top, uint32_t * seed);

/**
 * @brief Get the FPGA uptime in seconds.
 *
 * @param hw_top The HW top object.
 * @param uptime: The uptime
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_fpga_uptime(const enyx_hw_top * hw_top, uint32_t * uptime);

/**
 * @brief Get the FPGA serial ID.
 *
 * @param hw_top The HW top object.
 * @param serial_id: The serial ID
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_fpga_serial_id(const enyx_hw_top * hw_top,
                               enyx_hw_top_fpga_serial_id * serial_id);

/**
 * @brief Get the board temperature in Celsius degrees.
 *
 * @param hw_top The HW top object.
 * @param temp The temperature in celsius degrees
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_temperature(const enyx_hw_top * hw_top, double * temp);

/**
 * @brief Get the memory initialization status.
 * The result is a bitset, bit 0 is for memory 0 and so on, set to 1 when
 * memory is correcly initilized.
 *
 * @param hw_top The HW top object.
 * @param status A bitset of memory init status.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_memories_init_status(const enyx_hw_top * hw_top,
                                     uint32_t * status);

/**
 * @deprecated
 * Use @ref enyx_hw_top_get_firmware_identifier instead.
 *
 * @brief Get the unique identifier (firmware hash).
 *
 * @param hw_top The HW top object.
 * @param id The identifier.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_unique_identifier(const enyx_hw_top * hw_top, uint64_t * id)
ENYX_CORES_C_DEPRECATED("replaced by enyx_hw_top_get_firmware_identifier");

/**
 * @brief Get the firmware identifier.
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_firmware_identifier(const enyx_hw_top * hw_top,
                                    char * buffer, size_t size);

/**
 * @brief Get the stream interface count.
 *
 * @param hw_top The HW top object.
 * @param st_count The ST count.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_stream_count(const enyx_hw_top * hw_top, uint8_t * st_count);

/**
 * @brief Get the memory interface count.
 *
 * @param hw_top The HW top object.
 * @param mem_count The memory count.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_memory_count(const enyx_hw_top * hw_top, uint8_t * mem_count);

/**
 * @brief Get the clock interface count.
 *
 * @param hw_top The HW top object.
 * @param clock_count The clock count.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_clock_count(const enyx_hw_top * hw_top, uint8_t * clock_count);

/**
 * @brief Get the clock frequency in kHz.
 *
 * @param hw_top The HW top object.
 * @param clock_freq The clock frequency in kHz.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_clock_frequency(const enyx_hw_top * hw_top,
                                uint32_t * clock_freq);

/**
 * @brief Get the board MAC addresses.
 *
 * A number of MAC addresses are allocated for the board and available to use.
 * If more than one is available, they are incremented from the first one.
 * Ex. if the first one is 00:00:00:00:00:01 and ther are 3 MAC available,
 * the other ones will be 00:00:00:00:00:02 and 00:00:00:00:00:03.
 *
 * To retrieve the number of allocated MAC addresses, this function should be
 * called with a zero size, and it will return the number of MAC address.
 *
 * The return value will always be the MAC address count on success, which can
 * be greater (the @p addresses array will be incomplete) or lesser (the
 * @p addresses array will not be completely filled) than the provided size.
 *
 * @warning In order for the MAC addresses returned by this function to be
 * valid, the command <a href="../../../../../enyx-bsp/man/enyx-bsp-board.html#board-setup">enyx-bsp board setup</a>
 * have been run at least once. See the <a href="../../../../../enyx-bsp/contents/quickstart.html#first-time-firmware-installation-procedure">First time installation procedure</a>
 *
 * @param hw_top The HW top object.
 * @param addresses The MAC address array.
 * @param size The array size.
 * @return The available MAC address count on success, -1 on error (with errno
 * set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
ssize_t
enyx_hw_top_get_mac_addresses(const enyx_hw_top * hw_top,
                              enyx_mac_address addresses[], size_t size);

/**
 * @deprecated
 * Board serial ID retrieval through HW top is not supported anymore.
 *
 * @brief Get the board serial ID.
 *
 * @param hw_top The HW top object.
 * @param id The serial ID.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_board_serial_id(const enyx_hw_top * hw_top, uint32_t * id)
ENYX_CORES_C_DEPRECATED(
        "Board serial ID retrieval through HW top is not supported anymore");

/**
 * @brief Reset the whole FPGA board.
 *
 * @param hw_top The HW top object.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_reset_global(enyx_hw_top * hw_top);

/**
 * @brief Reset the FPGA modules.
 *
 * @param hw_top The HW top object.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_reset_module(enyx_hw_top * hw_top);

/**
 * @brief Reset the PHY interfaces.
 *
 * Interfaces set to 1 in the mask will be reset.
 *
 * @param hw_top The HW top object.
 * @param mask The interface mask.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_reset_phy(enyx_hw_top * hw_top, uint64_t mask);

/**
 * @brief Trigger an update of params from flash.
 *
 * Flash params include MAC address, MAC count and board serial ID.
 *
 * @param hw_top The HW top object.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_update_params(enyx_hw_top * hw_top);

/**
 * @brief Get the top name.
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 * @return 0 on success, -1 on error (with errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_top_name(const enyx_hw_top * hw_top,
                         char * buffer, size_t size);

/**
 * @brief Get the product ID.
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_product_id(const enyx_hw_top * hw_top,
                           char * buffer, size_t size);

/**
 * @brief Get the build machine hostname.
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_build_host(const enyx_hw_top * hw_top,
                         char * buffer, size_t size);

/**
 * @brief Get the build tool name and version.
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_build_tool(const enyx_hw_top * hw_top,
                             char * buffer, size_t size);

/**
 * @brief Get the FPGA device ID.
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_device_id(const enyx_hw_top * hw_top,
                          char * buffer, size_t size);

/**
 * @brief Get the board name (model).
 *
 * @param hw_top The HW top object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_hw_top_get_board_name(const enyx_hw_top * hw_top,
                           char * buffer, size_t size);

/**
 * @brief Create an HW top management object from a core subtree.
 *
 * @param subtree The core subtree to look up.
 * @return The HW top object.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
enyx_hw_top *
enyx_hw_top_create(enyx_hw_core * subtree);

/**
 * @brief Destroy a HW top object.
 *
 * @param hw_top The HW top object to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
void
enyx_hw_top_destroy(enyx_hw_top * hw_top);
