#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/hw_c/core.h>

/**
 * The main flash management library object.
 */
typedef struct enyx_flash enyx_flash;

/**
 * The flash memory section representation.
 *
 * Flash memory is split in different logical sections, used for various uses
 * (ex. firmware section, user data section...).
 *
 * This object is used to interact with this flash section
 * (read / write / get size).
 */
typedef struct enyx_flash_section enyx_flash_section;

/// Program flash section name
ENYX_CORES_C_SYMBOL_VISIBILITY
char const * ENYX_FLASH_SECTION_PROGRAM;

/// User flash section name
ENYX_CORES_C_SYMBOL_VISIBILITY
char const * ENYX_FLASH_SECTION_USER;

/// Params flash section name
ENYX_CORES_C_SYMBOL_VISIBILITY
char const * ENYX_FLASH_SECTION_PARAMS;

/**
 * @deprecated
 * Use @ref enyx_flash_get_sections(@ref ENYX_FLASH_SECTION_PROGRAM) instead.
 *
 * Get the flash memory program section object.
 *
 * @param flash The flash object to use.
 * @return The flash section object on success. NULL on failure (with @b errno
 * set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY const enyx_flash_section *
enyx_flash_get_section_pgm(enyx_flash const * flash)
ENYX_CORES_C_DEPRECATED("replaced by enyx_flash_get_sections");

/**
 * @deprecated
 * Use @ref enyx_flash_get_sections(@ref ENYX_FLASH_SECTION_USER) instead.
 *
 * Get the flash memory user section object.
 *
 * @param flash The flash object to use.
 * @return The flash section object on success. NULL on failure (with @b errno
 * set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY const enyx_flash_section *
enyx_flash_get_section_usr(enyx_flash const * flash)
ENYX_CORES_C_DEPRECATED("replaced by enyx_flash_get_sections");

/**
 * @deprecated
 * Use @ref enyx_flash_get_sections(@ref ENYX_FLASH_SECTION_PARAMS) instead and
 * take the first returned section.
 *
 * Get the flash memory params section object.
 *
 * @param flash The flash object to use.
 * @return The flash section object on success. NULL on failure (with @b errno
 * set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY const enyx_flash_section *
enyx_flash_get_section_prm(enyx_flash const * flash)
ENYX_CORES_C_DEPRECATED("replaced by enyx_flash_get_sections");

/**
 * Use @ref enyx_flash_get_sections(@ref ENYX_FLASH_SECTION_PARAMS) instead and
 * take the second returned section.
 *
 * Get the flash memory OTP section object.
 *
 * @param flash The flash object to use.
 * @return The flash section object on success. NULL on failure (with @b errno
 * set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY const enyx_flash_section *
enyx_flash_get_section_otp(enyx_flash const * flash)
ENYX_CORES_C_DEPRECATED("replaced by enyx_flash_get_sections");

/**
 * Read data from a flash memory section.
 *
 * @param section The flash section to read from.
 * @param data The byte array in which data will be dumped.
 * @param offset The offset in byte from the beginning of the section after
 * which data will be read.
 * @param size The size of the data array in bytes.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_read_section(const enyx_flash_section * section,
        uint8_t * data, size_t offset, size_t size);

/**
 * Write data to a flash memory section.
 *
 * @param section The flash section to write to.
 * @param data The byte array containing data to write.
 * @param offset The offset in byte from the beginning of the section after
 * which data will be written.
 * @param size The size of the data array in bytes.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_write_section(const enyx_flash_section * section,
        const uint8_t * data, size_t offset, size_t size);

/**
 * Get the flash section size.
 *
 * @param section The flash section object to use.
 * @param[out] size A pointer to the size on success.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_get_section_size(const enyx_flash_section * section, size_t * size);

/**
 * Get the flash section start address.
 *
 * @param section The flash section object to use.
 * @param[out] addr A pointer to the address on success.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_get_section_start_address(const enyx_flash_section * section, size_t * addr);

/**
 * Get the flash section name.
 *
 * @param section The flash section object to use.
 * @return The section name on success, NULL on error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY char const *
enyx_flash_get_section_name(enyx_flash_section const * section);

/**
 * Read data from a flash controller register.
 *
 * @param flash The flash object to use.
 * @param reg_addr The flash register address.
 * @param buffer The byte array in which data will be dumped.
 * @param size The size of the data array in bytes.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_read_register(enyx_flash const * flash, size_t reg_addr, uint8_t * buffer, size_t size);

/**
 * Write data to a flash controller register.
 *
 * @param flash The flash object to use.
 * @param reg_addr The flash register address.
 * @param data The byte array containing data to write.
 * @param size The size of the data array in bytes.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_write_register(enyx_flash * flash, size_t reg_addr, uint8_t * data, size_t size);

/**
 * Set the FPGA to reload its firmware upon next reboot.
 *
 * @param flash The flash object to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_set_fpga_reload_at_reboot(enyx_flash * flash);

/**
 * Reset the flash memory controller configuration.
 *
 * @param flash The flash object to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_reset_memory(enyx_flash * flash);

/**
 * @deprecated This function has no effect
 *
 * Enable or disable backup of remaining data from erased sections during
 * memory write.
 *
 * @param flash The flash object to use.
 * @param enable Whether to enable (@b true) or disable (@b false) the backup.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_set_erase_backup(enyx_flash * flash, bool enable)
ENYX_CORES_C_DEPRECATED("this function has no effect");

/**
 * Get the full flash memory size.
 *
 * @param flash The flash object to use.
 * @param[out] size A pointer to the size on success.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_get_memory_size(enyx_flash const * flash, size_t * size);

/**
 * Get the flash sector size.
 *
 * @param flash The flash object to use.
 * @param[out] size A pointer to the size on success.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_get_sector_size(enyx_flash const * flash, size_t * size);

/**
 * Check if firmware warm update is supported on this board.
 *
 * @param flash The flash object to use.
 * @param supported The warm update support
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_flash_is_warm_update_supported(enyx_flash const * flash, bool * supported);

/**
 * Get flash memory sections by name.
 *
 * @param flash The flash object to use.
 * @param name The flash section name or NULL to get all sections.
 * @param sections The sections to fill.
 * @param size The sections array size.
 * @return The count of matching sections on success, NULL on failure (with @b
 * errno set accordingly).
 *
 * @note This function can be used with a 0 @p size to count the flash sections
 * matching @p name (or all flash sections) by checking the return value.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_flash_get_sections(enyx_flash const * flash, char const * name,
                        enyx_flash_section const * sections[], size_t size);

/**
 * Create the flash management object from a core subtree.
 *
 * Enumerates the subtree until a flash_mng core is found and create a flash
 * management object from it.
 *
 * @param core The core to use.
 * @return The flash object on success. NULL on failure (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_flash *
enyx_flash_create(enyx_hw_core * core);

/**
 * Destroy a flash object.
 *
 * @param flash The flash object to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_flash_destroy(enyx_flash * flash);
