#pragma once

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/hw_c/core.h>

/**
 * @brief Values of supported data rates returned in the
 * @ref enyx_eth_if_get_supported_data_rates function.
 */
enum enyx_eth_if_data_rate {
    ENYX_ETH_IF_RATE_10M = 1 << 0,
    ENYX_ETH_IF_RATE_100M = 1 << 1,
    ENYX_ETH_IF_RATE_1000M = 1 << 2,
    ENYX_ETH_IF_RATE_10G = 1 << 3,
    ENYX_ETH_IF_RATE_25G = 1 << 4,
    ENYX_ETH_IF_RATE_40G = 1 << 5,
    ENYX_ETH_IF_RATE_100G = 1 << 6,
};

/**
 * @brief ETH IF management object.
 */
typedef struct enyx_eth_if enyx_eth_if;

/**
 * @brief Get the supported data rate as a bitfield of
 * @ref enyx_eth_if_data_rate values.
 *
 * @param eth_if The ETH IF object.
 * @return A bit field containing the data rates values or -1 on error (with
 * @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_supported_data_rates(enyx_eth_if const * eth_if);

/**
 * @brief Check if I2C is supported.
 *
 * @param eth_if The ETH IF object.
 * @return 1 if I2C is supported, 0 if not, or -1 on error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_i2c_support(enyx_eth_if const * eth_if);

/**
 * Get the I2C controller ID.
 *
 * @param eth_if The ETH IF object.
 * @param id The controller ID.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_i2c_controller_id(enyx_eth_if const * eth_if, uint8_t * id);

/**
 * @brief Get the I2C configuration.
 *
 * @param eth_if The ETH IF object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_i2c_configuration(enyx_eth_if const * eth_if,
                                  char * buffer, size_t size);

/**
 * @brief Get the interface clock frequency.
 *
 * @param eth_if The ETH IF object.
 * @param frequency The interface clock frequency in kHz.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_clock_frequency(enyx_eth_if const * eth_if,
                                uint32_t * frequency);

/**
 * @brief Get the interface data width.
 *
 * @param eth_if The ETH IF object.
 * @param data_width The interface data width.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_data_width(enyx_eth_if const * eth_if, uint32_t * data_width);

/**
 * @brief Check if this ETH IF provides a Linux netif.
 *
 * @param eth_if The ETH IF object.
 * @return 1 if enabled, 0 if not, or -1 on error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_is_netif_enabled(enyx_eth_if const * eth_if);

/**
 * @brief Get the host NIC ID.
 *
 * @param eth_if The ETH IF object.
 * @param id The host NIC ID.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_netif_id(enyx_eth_if const * eth_if, uint8_t * id);

/**
 * @brief Get the physical ID.
 *
 * @param eth_if The ETH IF object.
 * @param id The physical ID.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_physical_id(enyx_eth_if const * eth_if, uint32_t * id);

/**
 * @brief Get the physical name.
 *
 * @param eth_if The ETH IF object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_physical_name(enyx_eth_if const * eth_if,
                              char * buffer, size_t size);

/**
 * @brief Get the usage name.
 *
 * @param eth_if The ETH IF object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_usage_name(enyx_eth_if const * eth_if,
                           char * buffer, size_t size);

/**
 * @brief Get the RAW configuration.
 *
 * @param eth_if The ETH IF object.
 * @param buffer The buffer to fill.
 * @param size The buffer size.
 * @return 0 on success, -1 on  error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_eth_if_get_raw_configuration(enyx_eth_if const * eth_if,
                                  char * buffer, size_t size);

/**
 * Create the ETH IF subsystem from an eth_if core.
 *
 * @param core The core to use.
 * @return The subsystem on succes. NULL on failure (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
enyx_eth_if *
enyx_eth_if_create(enyx_hw_core * core);

/**
 * Destroy an ETH IF subsystem.
 *
 * @param eth_if The ETH IF to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
void
enyx_eth_if_destroy(enyx_eth_if * eth_if);
