#pragma once

#include <cstdint>
#include <memory>

#include <enyx/cores_c/uart/uart.h>

#include <enyx/hw/core.hpp>
#include <enyx/cores/macros.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_DEFAULT_DELETE(uart);

ENYX_CORES_NAMESPACE_BEGIN

namespace uart {

/**
 * Main UART subsystem object.
 *
 * This wrapper provides @b RAII of @b C @ref enyx_uart
 */
class uart {
public:
    /**
     * Construct an UART instance from a core subtree.
     *
     * Enumerates the subtree until a muart core is found and create an @b uart
     * object from it.
     *
     * @param subtree The core subtree to look up.
     * @throw system_error on failure
     */
    uart(enyx::hw::core const & subtree);

    /**
     * Get the UART core clock frequency.
     *
     * @return A result object containing the UART core clock frequency on
     * success or an error on failure.
     */
    result<std::uint32_t>
    get_clock_frequency() const noexcept;

    /**
     * Get the UART baudrate.
     *
     * @return A result object containing the UART baudrate on success or an
     * error on failure.
     */
    result<std::uint32_t>
    get_baudrate() const noexcept;

    /**
     * Check if the UART core RX fifo is empty.
     *
     * @return A result object containing @b true if the fifo is empty,
     * @b false if not, or an error on failure.
     */
    result<bool>
    is_rx_empty() const noexcept;

    /**
     * Check if the UART core TX fifo is full.
     *
     * @return A result object containing @b true if the fifo is full,
     * @b false if not, or an error on failure.
     */
    result<bool>
    is_tx_full() const noexcept;

    /**
     * Check if the UART RX lane is busy (data transfer in proress).
     *
     * @return A result object containing @b true if the lane is busy, @b false
     * if not, or an error on failure.
     */
    result<bool>
    is_rx_busy() const noexcept;

    /**
     * Check if the UART TX lane is busy (data transfer in proress).
     *
     * @return A result object containing @b true if the lane is busy, @b false
     * if not, or an error on failure.
     */
    result<bool>
    is_tx_busy() const noexcept;

    /**
     * Set the UART baudrate divisor.
     *
     * @param divisor The baudrate divisor to apply.
     * @return A result object containing an error on failure.
     */
    result<void>
    set_baudrate_divisor(std::uint16_t divisor) noexcept;

    /**
     * Clear the core RX fifo.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    clear_rx_fifo() noexcept;

    /**
     * Clear the core TX fifo.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    clear_tx_fifo() noexcept;

    /**
     * Read data from the UART core RX fifo.
     *
     * Read data until the RX fifo is empty, or the requested size is reached,
     * without waiting for new data.
     *
     * @param data A pointer to read the data to.
     * @param size The size of the @p data pointer.
     * @return A result object containing the amount of data read from the fifo
     * on success, or an error on failure.
     */
    result<std::size_t>
    read(std::uint8_t * data, std::size_t size) noexcept;

    /**
     * Try to read @p size data from the UART core RX fifo.
     *
     * Read data until the requested size is reached, or the given timeout is
     * reached when waiting for new data. On timeout, the returned read size
     * will be smaller than the requested size.
     *
     * @param data A pointer to read the data to.
     * @param size The size of the @p data pointer.
     * @param timeout_ms The maximum time to wait for new data.
     * @return A result object containing the amount of data read on success,
     * or an error on failure.
     */
    result<std::size_t>
    read_all(std::uint8_t * data, std::size_t size,
             std::uint64_t timeout_ms) noexcept;

    /**
     * Write data to the UART core TX fifo.
     *
     * Write data until the TX fifo is full, or all the provided data is
     * written, without waiting to try to send more data.
     *
     * @param data A pointer to the data to write.
     * @param size The size of the @p data pointer.
     * @return A result object containing the amount of data written to the
     * fifo on success, or an error on failure.
     */
    result<size_t>
    write(const std::uint8_t * data, std::size_t size) noexcept;

    /**
     * Try to write @p size data to the UART core TX fifo.
     *
     * Try to write data until all provided data is written, or the given
     * timeout is reached when waiting for fifo space. On timeout, the returned
     * written size will be smaller than the requested size.
     *
     * @param data A pointer to the data to write.
     * @param size The size of the @p data pointer.
     * @param timeout_ms The maximum time to wait for data to be sent.
     * @return A result object containing the amount of data written on
     * success, or an error on failure.
     */
    result<std::size_t>
    write_all(const std::uint8_t * data, std::size_t size,
              std::uint64_t timeout_ms) noexcept;

    /**
     * Access to the underlying @b C @ref enyx_uart handle.
     *
     * @return The @b C handle.
     */
    ::enyx_uart const *
    handle() const noexcept;

    /**
     * Access to the underlying @b C @ref enyx_uart handle.
     *
     * @return The @b C handle.
     */
    ::enyx_uart *
    handle() noexcept;

private:
    enyx::hw::core subtree_root_;
    std::unique_ptr<::enyx_uart> uart_;
};

} /* namespace uart */

ENYX_CORES_NAMESPACE_END

#include "uart.ipp"
