#pragma once

#include <enyx/cores_c/net_interface/net_interface.h>

#include <vector>
#include <memory>

#include <enyx/hw/core.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/types.hpp>
#include <enyx/cores/result.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_net_interfaces>
{
    void
    operator()(::enyx_net_interfaces * ptr) const
    {
        ::enyx_net_interfaces_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace net_interface {

/// @copydoc enyx_net_interface_ip_config
class ip_config final
{
public:
    /**
     * Construct an uninitialized IP configuation.
     */
    ip_config() noexcept;

    /**
     * Construct a IP configuration from a @b C ip config object.
     *
     * @param c_config The @b C ip config object to construct from
     */
    ip_config(::enyx_net_interface_ip_config const& c_config) noexcept;

    /**
     * Construct a ip configuration from its @p ip, and @p mask.
     *
     * @note No validation whatsoever is performed on the mask.
     *
     * @param ip The ip
     * @param mask The associated mask
     */
    ip_config(ipv4_address const& ip,
              ipv4_address const& mask) noexcept;

    /**
     * Construct a ip configuration from its @p ip, and netmask @p prefix.
     *
     * @note No validation whatsoever is performed on the prefix.
     *
     * @param ip The IP address
     * @param prefix The associated mask
     */
    ip_config(ipv4_address const& ip,
              std::uint8_t prefix);

    /**
     * Get the IPv4 address of this configuration.
     *
     * @return The IPv4 address
     */
    ipv4_address
    get_ip() const noexcept;

    /**
     * Get the subnetwork mask of this configuration.
     *
     * @return The mask
     */
    ipv4_address
    get_mask() const noexcept;

    /**
     * Get the subnetwork prefix of this configuration.
     *
     * @return The prefix
     */
    std::uint8_t
    get_prefix() const noexcept;

    /**
     * Convert a @b C++ object to a @b C object.
     *
     * @return The @b C object
     */
    operator ::enyx_net_interface_ip_config *() noexcept;

    /**
     * Convert a @b C++ object to a @b C object.
     *
     * @return The @b C object
     */
    operator ::enyx_net_interface_ip_config const*() const noexcept;

private:
    ::enyx_net_interface_ip_config config_;
};

/**
 * Print an @p ip configuration into a stream @p out.
 *
 * Output format: "ip <ip> netmask <netmask>"
 *
 * @param out The stream to use
 * @param ip_config The ip_config to output
 * @return A reference to the output stream @p out
 */
std::ostream &
operator<<(std::ostream & out, ip_config const& ip_config);

/// @copydoc enyx_net_interface_vlan_pcp
enum class vlan_pcp
{
    /** BE - Best Effort */
    BEST_EFFORT = ENYX_INTERFACE_VLAN_PCP_BEST_EFFORT,
    /** BK - Background */
    BACKGROUND = ENYX_INTERFACE_VLAN_PCP_BACKGROUND,
    /** EE - Excellent Effort */
    EXCELLENT_EFFORT = ENYX_INTERFACE_VLAN_PCP_EXCELLENT_EFFORT,
    /** CA - Critical Applications */
    CRITICAL_APPLICATION = ENYX_INTERFACE_VLAN_PCP_CRITICAL_APPLICATION,
    /** VI = Video, < 100 ms latency and jitter */
    VIDEO = ENYX_INTERFACE_VLAN_PCP_VIDEO,
    /** VO - Voice, < 10 ms latency and jitter */
    VOICE = ENYX_INTERFACE_VLAN_PCP_VOICE,
    /** IC - Internetwork Control */
    INTERNETWORK_CONTROL = ENYX_INTERFACE_VLAN_PCP_INTERNETWORK_CONTROL,
    /** NC - Network Control */
    NETWORK_CONTROL = ENYX_INTERFACE_VLAN_PCP_NETWORK_CONTROL,
};

/**
 * Print a @p vlan pcp into a stream @p out.
 *
 * For example "network_control".
 *
 * @param out The stream to use
 * @param vlan_pcp The vlan pcp to output
 * @return A reference to the output stream @p out
 */
std::ostream &
operator<<(std::ostream & out, vlan_pcp const& vlan_pcp);

/**
 * Scan a @p vlan_pcp from a stream @p in.
 *
 * For example "excellent_effort"
 *
 * @param in The stream to use
 * @param vlan_pcp The vlan pcp to scan
 * @return A reference to the input stream @p in
 */
std::istream &
operator>>(std::istream & in, vlan_pcp & vlan_pcp);

/// @copydoc enyx_net_interface_vlan_config
class vlan_config final
{
public:
    /// The vlan id
    using id = std::uint16_t;

    // The vlan pcp
    using pcp = vlan_pcp;

    // The vlan DEI
    using dei = bool;

public:
    /**
     * Construct an uninitialized vlan configuation.
     */
    vlan_config() noexcept;

    /**
     * Construct a vlan configuration from a @b C vlan config object.
     *
     * @param c_config The @b C vlan config object to construct from
     */
    vlan_config(::enyx_net_interface_vlan_config const& c_config) noexcept;

    /**
     * Construct a vlan configuration from its @p id, and
     * optional @p pcp & @p dei.
     *
     * @param id The id of the vlan (0 means unused)
     * @param pcp The vlan priority code
     * @param dei The vlan drop eligible indicator
     */
    vlan_config(id id,
                pcp pcp = pcp::BEST_EFFORT,
                dei dei = false) noexcept;

    /**
     * Get the vlan ID.
     *
     * @return The id
     */
    id
    get_id() const noexcept;

    /**
     * Get the vlan Priority Code Point.
     *
     * @return The pcp
     */
    pcp
    get_pcp() const noexcept;

    /**
     * Get the vlan Drop Eligible Indicator.
     *
     * @return The dei
     */
    dei
    get_dei() const noexcept;

    /**
     * Convert a @b C++ object to a @b C object.
     *
     * @return The @b C object
     */
    operator ::enyx_net_interface_vlan_config * () noexcept;

    /**
     * Convert a @b C++ object to a @b C object.
     *
     * @return The @b C object
     */
    operator ::enyx_net_interface_vlan_config const* () const noexcept;

private:
    ::enyx_net_interface_vlan_config config_;
};

/**
 * Compare vlan config @p a & @p b for equality.
 *
 * @param a The first vlan config
 * @param b The second vlan config
 * @return true if both @p a & @p b are equal, false otherwise
 */
bool
operator==(vlan_config const& a, vlan_config const& b);

/**
 * Compare vlan config @p a & @p b for equality.
 *
 * @param a The first vlan config
 * @param b The second vlan config
 * @return true if both @p a & @p b are different, false otherwise
 */
bool
operator!=(vlan_config const& a, vlan_config const& b);

/**
 * Print a @p vlan config into a stream @p out.
 *
 * For example "id 1 pcp best_effort dei false".
 *
 * @param out The stream to use
 * @param vlan_config The vlan config to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, vlan_config const& vlan_config);

/// @copydoc enyx_net_interface_statistics
using statistics = ::enyx_net_interface_statistics;

/**
 *  Manipulate hardware virtual net_interfaces.
 *
 *  This wrapper provides @b RAII of @b C @ref enyx_net_interfaces.
 */
class net_interfaces final
{
public:
    /**
     * Construct the NET_INTERFACE module
     *
     * This module configures the following hw::core:
     *
     * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     * * tcp_multi_stack (from TCP_STD 2.x)
     * * udp_multi_stack (from UDP_STD 2.x)
     *
     * @param core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    net_interfaces(enyx::hw::core const& core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Get the number of net_interfaces in this subsystem.
     *
     * @return The number of net_interfaces
     */
    std::uint16_t
    count() const noexcept;

    /**
     * Test if an net_interface is configured.
     *
     * @note tcp_multi_stack and udp_multi_stack net_interfaces are configured
     * when their mac address is set.
     *
     * @param interface_id The net_interface id of the selected net_interface
     * @return True if the net_interface is up, false otherwise
     */
    bool
    configured(std::uint16_t interface_id) const noexcept;

    /**
     * Mark an net_interface as configured. While this is a hardware setting, it is
     * mostly an indication that the net_interface should not be modified
     * afterwards.
     *
     * @note On tcp_multi_stack and udp_multi_stack, this function is not
     * implemented.
     *
     * @param interface_id The net_interface id of the selected net_interface
     * @return a result object containing an error on failure
     */
    result<void>
    configure(std::uint16_t interface_id) noexcept;

    /**
     * Unmark an net_interface as configured. While this is a hardware setting, it
     * is mostly an indication that the net_interface may be modified afterwards.
     *
     * @note On tcp_multi_stack and udp_multi_stack, this resets the net_interface.
     *
     * @param interface_id The net_interface id of the selected net_interface
     * @return a result object containing an error on failure
     */
    result<void>
    unconfigure(std::uint16_t interface_id) noexcept;

    /**
     * Get the mac address of this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @return a result object containing an error or the mac address
     */
    result<mac_address>
    get_mac(std::uint16_t interface_id) const noexcept;

    /**
     * Set the mac address for this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @param mac_address The mac address to set
     * @return a result object containing an error on failure
     *
     * @note The MAC address used here should be in the range given by @ref
     * hw_top::hw_top::get_mac_addresses
     */
    result<void>
    set_mac(std::uint16_t interface_id,
            mac_address const & mac_address) noexcept;

    /**
     * Get the address of this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @return a result object containing an error or the address
     */
    result<ip_config>
    get_ip_config(std::uint16_t interface_id) const noexcept;

    /**
     * Set the address for this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @param address The address the set
     * @return a result object containing an error on failure
     */
    result<void>
    set_ip_config(std::uint16_t interface_id,
                  ip_config const & address) noexcept;

    /**
     * Get the gateway of this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @return a result object containing an error or the gateway
     */
    result<ipv4_address>
    get_gateway(std::uint16_t interface_id) const noexcept;

    /**
     * Set the gateway for this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @param address The address the set
     * @return a result object containing an error on failure
     */
    result<void>
    set_gateway(std::uint16_t interface_id,
                ipv4_address const & address) noexcept;

    /**
     * Set the vlan configuration for this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @param vlan The vlan to set
     * @return a result object containing an error on failure
     */
    result<void>
    set_vlan(std::uint16_t interface_id,
             vlan_config const & vlan) noexcept;

    /**
     * Get the vlan configuration of this net_interface.
     *
     * @param interface_id The id of the net_interface
     * @return a result object containing an error or the vlan
     */
    result<vlan_config>
    get_vlan(std::uint16_t interface_id) const noexcept;

    /**
     * Get this subsystem's MTU (aka Ethernet MTU).
     *
     * @return A result object containing the MTU or an error
     */
    result<std::uint16_t>
    get_mtu() const noexcept;

    /**
     * @copybrief enyx_net_interfaces_get_statistics
     *
     * @return A result object containing the statistics on success or an error
     *         on failure
     */
    result<statistics>
    get_statistics() const noexcept;

    /**
     * Get the underlying C handle
     *
     * @return The C handle
     */
    enyx_net_interfaces *
    handle() const noexcept;

private:
    enyx::hw::core net_interfaces_core_;
    std::unique_ptr<::enyx_net_interfaces> handle_;
};

} /* namespace net_interface */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/net_interface/net_interface.ipp>
