#pragma once
#include <string>
#include <sstream>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/hw_top/hw_top.hpp>

ENYX_CORES_NAMESPACE_BEGIN

namespace hw_top {
namespace mocking {

/**
 * Wrap a generated mocked hw_top core with major version 1
 *
 * This mocking should be used as base for any hw_top mock: The hw_top API
 * needs working registers for all strings on creation and this class provides
 * compatible defaults.
 *
 * @tparam MockGeneratedHwTop The generated mock to inherit from
 */
template<class MockGeneratedHwTop>
struct hw_top_1_mock : public MockGeneratedHwTop {
    hw_top_1_mock()
        : MockGeneratedHwTop()
        , firmware_name{"taupe"}
        , product_id{"poc"}
        , build_host{"localhost"}
        , build_tool{"1970.12"}
        , fpga_device{"mcu-sea-horse"}
        , board_model{"fpb3"}
        , firmware_build_date{4242}
        , firmware_identifier{0xdeadbeef}
        , fpga_serial{}
        , global_reset_called{false}
        , module_reset_called{false}
        , phy_reset_called{false}
        , update_params_called{false}
    {
        this->get_top_name_string = [this] () {
            return this->firmware_name;
        };

        this->get_product_id_string = [this] () {
            return this->product_id;
        };

        this->get_hostname_string = [this] {
            return this->build_host;
        };

        this->get_tool_version_string = [this] {
            return this->build_tool;
        };

        this->get_device_id_string = [this] {
            return this->fpga_device;
        };

        this->get_board_name_string = [this] {
            return this->board_model;
        };

        this->build_date.on_read = [this] () -> std::uint64_t {
            return this->firmware_build_date;
        };

        this->identifier.on_read = [this] () {
            return this->firmware_identifier;
        };

        this->fpga_serial_id_0.on_read = [this] {
            uint32_t serial_0;
            std::memcpy(&serial_0,
                        this->fpga_serial.value,
                        sizeof(serial_0));
            return serial_0;
        };

        this->fpga_serial_id_1.on_read = [this] {
            uint32_t serial_1;
            std::memcpy(&serial_1,
                        this->fpga_serial.value + 4,
                        sizeof(serial_1));
            return serial_1;
        };

        this->fpga_serial_id_2.on_read = [this] {
            uint32_t serial_2;
            std::memcpy(&serial_2,
                        this->fpga_serial.value + 8,
                        sizeof(serial_2));
            return serial_2;
        };

        this->on_all_cmd_global_reset_exec = [this] () {
            this->all_cmd_global_reset_exec.set(0);
            this->global_reset_called = true;
        };

        this->on_all_cmd_module_reset_exec = [this] () {
            this->all_cmd_module_reset_exec.set(0);
            this->module_reset_called = true;
        };

        this->on_all_cmd_phy_if_reset_exec = [this] () {
            this->all_cmd_phy_if_reset_exec.set(0);
            this->phy_reset_called = true;
        };

        this->on_all_cmd_update_param_exec = [this] () {
            this->all_cmd_update_param_exec.set(0);
            this->update_params_called = true;
        };
    }

    std::string firmware_name;
    std::string product_id;
    std::string build_host;
    std::string build_tool;
    std::string fpga_device;
    std::string board_model;
    std::uint64_t firmware_build_date;
    std::uint64_t firmware_identifier;
    ::enyx_hw_top_fpga_serial_id fpga_serial;

    bool global_reset_called;
    bool module_reset_called;
    bool phy_reset_called;
    bool update_params_called;
};


/**
 * Wrap a generated mocked hw_top core with major version 2
 *
 * This mocking should be used as base for any hw_top mock: The hw_top API
 * needs the param string to be set properly in order to function and this
 * class provides a compatible default.
 *
 * @tparam MockGeneratedHwTop The generated mock to inherit from
 */
template<class MockGeneratedHwTop>
struct hw_top_2_mock : public MockGeneratedHwTop {
    hw_top_2_mock()
        : MockGeneratedHwTop()
        , firmware_name{"taupe"}
        , product_id{"poc"}
        , build_host{"localhost"}
        , build_tool{"1970.12"}
        , fpga_device{"mcu-sea-horse"}
        , board_model{"fpb3"}
        , firmware_build_date{0}
        , firmware_identifier{"0xdeadbeef"}
        , fpga_serial{}
        , global_reset_called{false}
        , module_reset_called{false}
        , phy_reset_called{false}
        , update_params_called{false}
    {
        this->get_firmware_param_string = [this] {
            std::ostringstream oss;
            oss << "TOP_NAME="  << firmware_name << "\n"
                << "PRODUCT_ID="  << product_id << "\n"
                << "HOSTNAME="  << build_host << "\n"
                << "TOOL_VERSION="  << build_tool << "\n"
                << "DEVICE_ID="  << fpga_device << "\n"
                << "BOARD_NAME="  << board_model << "\n"
                << "BUILD_DATE="  << firmware_build_date << "\n"
                << "IDENTIFIER="  << firmware_identifier;
            return oss.str();
        };

        this->fpga_serial_id_0.on_read = [this] {
            uint32_t serial_0;
            std::memcpy(&serial_0,
                        this->fpga_serial.value,
                        sizeof(serial_0));
            return serial_0;
        };

        this->fpga_serial_id_1.on_read = [this] {
            uint32_t serial_1;
            std::memcpy(&serial_1,
                        this->fpga_serial.value + 4,
                        sizeof(serial_1));
            return serial_1;
        };

        this->fpga_serial_id_2.on_read = [this] {
            uint32_t serial_2;
            std::memcpy(&serial_2,
                        this->fpga_serial.value + 8,
                        sizeof(serial_2));
            return serial_2;
        };

        this->on_resets_global_reset_exec = [this] () {
            this->resets_global_reset_exec.set(0);
            this->global_reset_called = true;
        };

        this->on_resets_module_reset_exec = [this] () {
            this->resets_module_reset_exec.set(0);
            this->module_reset_called = true;
        };

        this->on_resets_phy_if_reset_exec = [this] (std::uint64_t) {
            this->resets_phy_if_reset_exec.set(0);
            this->phy_reset_called = true;
        };

        this->on_update_param_exec = [this] () {
            this->update_param_exec.set(0);
            this->update_params_called = true;
        };
    }

    std::string firmware_name;
    std::string product_id;
    std::string build_host;
    std::string build_tool;
    std::string fpga_device;
    std::string board_model;
    std::uint64_t firmware_build_date;
    std::string firmware_identifier;
    ::enyx_hw_top_fpga_serial_id fpga_serial;

    bool global_reset_called;
    bool module_reset_called;
    bool phy_reset_called;
    bool update_params_called;
};

}
}

ENYX_CORES_NAMESPACE_END
