ENYX_CORES_NAMESPACE_BEGIN

namespace flash {

namespace {

inline static ::enyx_flash *
create_flash(enyx::hw::core & core)
{
    auto flash = ::enyx_flash_create(core.handle());
    if (flash == nullptr)
        throw std::system_error{errno, std::generic_category(),
            "Failed to create flash management subsystem"};

    return flash;
}

inline static enyx::result<std::vector<flash::section>>
get_sections_char(flash const & flash, char const * name)
{
    ssize_t section_count = ::enyx_flash_get_sections(flash.handle(), name,
                                                      NULL, 0);
    if (section_count < 0)
        return std::error_code{errno, std::generic_category()};

    std::vector<::enyx_flash_section const *> c_sections(section_count);
    if (::enyx_flash_get_sections(flash.handle(), name,
                                  c_sections.data(), c_sections.size()) < 0)
        return std::error_code{errno, std::generic_category()};

    std::vector<flash::section> sections;
    for (auto c_section: c_sections)
        sections.push_back(flash::section{c_section});

    return sections;
}

} // namespace

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
inline
flash::flash(enyx::hw::core const & core)
    : subtree_root_(core)
    , flash_(create_flash(subtree_root_))
    , pgm_(::enyx_flash_get_section_pgm(handle()))
    , usr_(::enyx_flash_get_section_usr(handle()))
    , prm_(::enyx_flash_get_section_prm(handle()))
    , otp_(::enyx_flash_get_section_otp(handle()))
{ }
#pragma GCC diagnostic pop

inline enyx::result<void>
flash::read_register(size_t reg_addr, uint8_t * buffer, size_t size) const noexcept
{
    if (::enyx_flash_read_register(handle(), reg_addr, buffer, size))
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<void>
flash::write_register(size_t reg_addr, uint8_t * data, size_t size) noexcept
{
    if (::enyx_flash_write_register(handle(), reg_addr, data, size))
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<void>
flash::set_fpga_reload_at_reboot() noexcept
{
    if (::enyx_flash_set_fpga_reload_at_reboot(handle()))
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<void>
flash::reset_memory() noexcept
{
    if (::enyx_flash_reset_memory(handle()))
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<void>
flash::set_erase_backup(bool enable) noexcept
{
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
    if (::enyx_flash_set_erase_backup(handle(), enable))
#pragma GCC diagnostic pop
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<size_t>
flash::get_memory_size() const noexcept
{
    size_t size;
    if (::enyx_flash_get_memory_size(handle(), &size) != 0) {
        return std::error_code{errno, std::generic_category()};
    }

    return enyx::result<size_t>(size);
}

inline enyx::result<size_t>
flash::get_sector_size() const noexcept
{
    size_t size;
    if (::enyx_flash_get_sector_size(handle(), &size) != 0) {
        return std::error_code{errno, std::generic_category()};
    }

    return enyx::result<size_t>(size);
}

inline enyx::result<bool>
flash::is_warm_update_supported() const noexcept
{
    bool supported;
    if (::enyx_flash_is_warm_update_supported(handle(), &supported) != 0) {
        return std::error_code{errno, std::generic_category()};
    }

    return enyx::result<bool>(supported);
}

inline enyx::result<std::vector<flash::section>>
flash::get_sections(std::string const & name) const noexcept
{
    return get_sections_char(*this, name.c_str());
}

inline enyx::result<std::vector<flash::section>>
flash::get_sections() const noexcept
{
    return get_sections_char(*this, nullptr);
}

inline flash::section const &
flash::pgm() const noexcept
{
    return pgm_;
}

inline flash::section const &
flash::usr() const noexcept
{
    return usr_;
}

inline flash::section const &
flash::prm() const noexcept
{
    return prm_;
}

inline flash::section const &
flash::otp() const noexcept
{
    return otp_;
}

inline enyx_flash const *
flash::handle() const noexcept
{
    return flash_.get();
}

inline enyx_flash *
flash::handle() noexcept
{
    return flash_.get();
}

inline
flash::section::section(const enyx_flash_section * section)
    :section_(section)
{
    if (section_ == nullptr)
        throw std::system_error(errno, std::generic_category(),
                "Failed to create section object");
}

inline enyx::result<void>
flash::section::read(uint8_t * data, size_t offset, size_t size) const noexcept
{
    if (::enyx_flash_read_section(section_, data, offset, size) < 0)
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<void>
flash::section::write(const uint8_t * data, size_t offset, size_t size) const
noexcept
{
    if (::enyx_flash_write_section(section_, data, offset, size) < 0)
        return std::error_code{errno, std::generic_category()};

    return enyx::result<void>();
}

inline enyx::result<size_t>
flash::section::start_address() const noexcept
{
    size_t addr;
    if (::enyx_flash_get_section_start_address(section_, &addr) != 0) {
        return std::error_code{errno, std::generic_category()};
    }

    return enyx::result<size_t>(addr);
}

inline enyx::result<uint32_t>
flash::section::size() const noexcept
{
    size_t size;
    if (::enyx_flash_get_section_size(section_, &size) != 0) {
        return std::error_code{errno, std::generic_category()};
    }

    return enyx::result<uint32_t>(size);
}

inline enyx::result<std::string>
flash::section::name() const noexcept
{
   char const * name = ::enyx_flash_get_section_name(section_);
   if (name == nullptr)
        return std::error_code{errno, std::generic_category()};

   return name;
}

} /* namespace flash */

ENYX_CORES_NAMESPACE_END
