#pragma once

#include <enyx/cores_c/data_stream/enyx_hw_source.h>

#include <cerrno>
#include <system_error>

#include <enyx/hw/a2c_stream.hpp>

#include <enyx/cores/data_stream/source.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_NAMESPACE_BEGIN
namespace data_stream {

/**
 *  C++ wrapper around @b C @ref enyx_data_stream_hw_source_poll_once
 *
 *  Basic usage:
 *  @code
 *  // Create the data handler.
 *  // It could be a stateful functor with states as well.
 *  auto on_data = [](std::uint8_t const * data,
 *                    std::uint32_t size,
 *                    void const * metadata) {
 *      std::cout << "Received " << size << " bytes-long message" << std::endl;
 *  };
 *
 *  // Initialize the HW source with an a2c stream and a data handler.
 *  enyx::data_stream::hw_source hw_source{a2c_stream, on_data};
 *
 *  // Poll for messages until exit is requested.
 *  while (! is_exit_requested)
 *    hw_source.poll_once();
 *  @endcode
 */
class hw_source {
public:
    /**
     *  Constructs a data_stream source from an enyx hw a2c @b stream calling
     *  @b handler.
     *
     *  @tparam Handler The handler type
     *  @param stream The enyx hw a2c stream to use
     *  @param handler The handler to call on data
     */
    template<typename Handler>
    hw_source(hw::a2c_stream & stream, Handler && handler);

    /**
     * Poll the stream for data.
     *
     * When data is received, the @b handler given at construction is called
     * and this function returns.
     *
     * @returns @b 1 if a message has been received otherwise @b 0.
     */
    result<int>
    poll_once() noexcept;

private:
    hw::a2c_stream & stream_;
    source_ptr next_source_;
};

} /* namespace data_stream */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/data_stream/hw_source.ipp>
