#pragma once

#include <enyx/cores_c/arp/arp.h>

#include <vector>
#include <memory>

#include <enyx/hw/core.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores/types.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_arp>
{
    void
    operator()(::enyx_arp * ptr) const
    {
        ::enyx_arp_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace arp {

/**
 * Simple type for adding/reading ARP entries
 */
struct arp_entry
{
    /// IPv4 Address of the entry
    ipv4_address ipv4;
    /// MAC Address of the entry
    mac_address mac;
};

/**
 * Manipulate hardware arp subsystem.
 *
 * This wrapper provides @b RAII of @b C @ref enyx_arp
 */
class arp
{
public:
    /**
     * Construct the ARP module
     *
     * This module configures the following hw::core:
     *
     * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     * * tcp_multi_stack (from TCP_STD 2.x)
     * * udp_multi_stack (from UDP_STD 2.x)
     *
     * @param arp_core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    arp(enyx::hw::core const& arp_core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Test if the ARP server feature is available on this subsystem.
     *
     * @return true if it is available
     */
    bool
    is_server_available() const noexcept;

    /**
     * Test if the ARP client feature is available on this subsystem.
     *
     * @return true if it is available
     */
    bool
    is_client_available() const noexcept;

    /**
     * Test if the ARP server is enabled on this subsystem.
     *
     * @note Only available if @ref is_server_available is true.
     *
     * @return true if it is enabled
     */
    bool
    is_server_enabled() const noexcept;

    /**
     * Enable the ARP server
     *
     * @note Only available if @ref is_server_available is true.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_server() noexcept;

    /**
     * Disable the ARP server
     *
     * @note Only available if @ref is_server_available is true.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_server() noexcept;

    /**
     * Get the current timeout (in miliseconds) for ARP replies.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @return A result object containing the timeout on success or an error on
     * failure
     */
    result<std::uint16_t>
    get_client_timeout() const noexcept;

    /**
     * Set the timeout (in miliseconds) for ARP replies.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    set_client_timeout(std::uint16_t timeout_ms) noexcept;

    /**
     * Get the current lifetime (in seconds) of dynamic ARP entries.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @return A result object containing the lifetime on success or an error
     * on failure
     */
    result<std::uint16_t>
    get_dynamic_lifetime() const noexcept;

    /**
     * Set the lifetime (in seconds) of dynamic ARP entries
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    set_dynamic_lifetime(std::uint16_t lifetime_s) noexcept;

    /**
     * Get the maximum number of entries in the ARP table.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @return The number of entries.
     */
    std::uint16_t
    table_get_max_entry_count() const noexcept;

    /**
     * Add a new static entry to the ARP table.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @param entry The entry to add
     * @return A result object containing an error on failure
     */
    result<void>
    table_add_entry(arp_entry const & entry) noexcept;

    /**
     * Get the mac address associated with an IP address in the ARP table.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @param ipv4 The IP address
     * @return A result object containing the mac addess on success or an error
     * on falure.
     */
    result<mac_address>
    table_find_mac(ipv4_address ipv4) noexcept;

    /**
     * Get an entry from the ARP table.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @param entry_id The index of the required entry.
     * @return The entry on success or an error on failure.
     */
    result<arp_entry>
    table_get_entry(std::uint16_t entry_id) const noexcept;

    /**
     * Delete an entry from the ARP table.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @param ipv4 The IPv4 address indicating the entry.
     * @return A result object containing an error on failure.
     */
    result<void>
    table_delete_entry(ipv4_address ipv4) noexcept;

    /**
     * Delete all dynamic entries from the ARP table.
     *
     * @note Only available if @ref is_client_available is true.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    table_flush_dynamic_entries() noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_arp *
    handle() const noexcept;

private:
    enyx::hw::core arp_core_;
    std::unique_ptr<::enyx_arp> handle_;
};

} /* namespace arp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/arp/arp.ipp>
