/*
 * HFP rx implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_rx_dev.h"

#include <linux/version.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/delay.h>

#include "enyx_hfp_common.h"

/* A 8-bit r/w 8 register: reset */
#define HFP_RX_RESET                        HFP_REG_ADDR(4, 0)

/* A 16-bit r/o register: cache line shift min */
#define HFP_RX_CACHE_LINE_SHIFT_MIN         HFP_REG_ADDR(5, 0)

/* A 16-bit r/o register: cache line shift max */
#define HFP_RX_CACHE_LINE_SHIFT_MAX         HFP_REG_ADDR(5, 2)

/* A 8-bit r/o register: channel count */
#define HFP_RX_DEVICE_COUNT                 HFP_REG_ADDR(6, 0)

/* A 16-bit r/o register: channel mtu */
#define HFP_RX_MTU                          HFP_REG_ADDR(6, 2)

/* A 16-bit r/o register: buffer size shit max */
#define HFP_RX_BUFFER_SIZE_SHIFT_MAX        HFP_REG_ADDR(7, 0)

/* A 32-bit r/o register: ctrl offset base address (dword) */
#define HFP_RX_CTRL_BASE_OFFSET_DW          HFP_REG_ADDR(8, 0)

/* A 32-bit r/o register: ctrl increment */
#define HFP_RX_CTRL_INCREMENT_DW            HFP_REG_ADDR(9, 0)

/* A 8-bit r/w register: cpu cache line shift */
#define HFP_RX_CACHE_LINE_SHIFT             HFP_REG_ADDR(10, 0)

/* A 8-bit r/w register: cmd exec */
#define HFP_RX_CMD_EXEC                     HFP_REG_ADDR(11, 0)

/* A 16-bit r/w register: cmd device id */
#define HFP_RX_CMD_DEVICE_ID                HFP_REG_ADDR(11, 2)

/* A 16-bit w/o register: cmd interrupt id */
#define HFP_RX_CMD_INTERRUPT_ID             HFP_REG_ADDR(12, 0)

/* A 32-bit w/o register: cmd buffer size shift */
#define HFP_RX_CMD_BUFFER_SIZE_SHIFT        HFP_REG_ADDR(13, 0)

/* A 64-bit w/o register: cmd buffer dma addr */
#define HFP_RX_CMD_BUFFER_ADDR              HFP_REG_ADDR(14, 0)

/* A 64-bit w/o register: cmd ptrs dma addr */
#define HFP_RX_CMD_PTR_ADDR                 HFP_REG_ADDR(16, 0)

/* A 8-bit w/o register: info device id */
#define HFP_RX_INFO_DEVICE_ID               HFP_REG_ADDR(50, 0)

/* A 8-bit w/o register: info usage str index */
#define HFP_RX_INFO_USAGE_STR_INDEX         HFP_REG_ADDR(50, 2)

/* A 32-bit r/o register: stats packet count */
#define HFP_RX_INFO_PACKET_COUNT            HFP_REG_ADDR(51, 0)

/* A 32-bit r/o register: stats backpressure count */
#define HFP_RX_INFO_BACKPRESSURE_COUNT      HFP_REG_ADDR(52, 0)

/* A 8-bit r/o register: info device */
#define HFP_RX_INFO_PAGE_READY              HFP_REG_ADDR(53, 0)

/* A 32-bit r/o register: usage str */
#define HFP_RX_INFO_USAGE_STR               HFP_REG_ADDR(54, 0)

/* A 32 bit r/o register: fifo overrun error count */
#define HFP_RX_INFO_FIFO_OVERRUN_COUNT          HFP_REG_ADDR(55, 0)

/* A 32 bit r/o register: errors */
#define HFP_RX_DEVICE_PACKET_COUNT_ERRORS   HFP_REG_ADDR(56, 0)

/* A 32-bit w/o register: ctrl rd ptr */
#define HFP_RX_CTRL_RD_PTR                  HFP_REG_ADDR(0, 0)

/* A 32-bit w/o register: ctrl interrupt mask */
#define HFP_RX_CTRL_INTERRUPT_MASK          HFP_REG_ADDR(1, 0)

/* A 16 bit r/o register: Current channel MTU size */
#define HFP_RX_DEVICE_MTU                   HFP_REG_ADDR(57, 0)

/* A 32 bit r/o register: truncated packet count (errors) */
#define HFP_RX_DEVICE_PACKET_COUNT_TRUNCATED   HFP_REG_ADDR(58, 0)

struct enyx_hfp_rx_coherent_buffer
{
    size_t size;
    void * virt_addr;
    dma_addr_t bus_addr;
};

struct enyx_hfp_rx_channel
{
    uint16_t channel_id;

    int irq;
    int int_bus_index;
    size_t int_mask_offset;

    size_t buffer_size_shift;
    size_t default_buffer_size_shift;
    struct enyx_hfp_rx_coherent_buffer data_buffer;
    struct enyx_hfp_rx_coherent_buffer ptrs_buffer;
    uint32_t buffer_offset;
    size_t buffer_rd_ptr_offset;
    uint32_t buffer_used_size;
    uint16_t mtu;
};

struct enyx_hfp_rx_msg
{
    uint32_t data_size : 16;
    uint32_t : 16;
    uint32_t unused[7];
    uint8_t data[];
};

struct enyx_hfp_rx_impl {
    struct enyx_hfp_device * parent;

    uint32_t channel_count;
    uint16_t global_mtu;
    size_t coherency_line_size;

    struct enyx_hfp_irq_list * irq_list;

    /* Ctrl */
    size_t ctrl_offset;
    size_t ctrl_increment;

    /* DMA Buffers */
    size_t buffers_size_shift;

    struct device * dma;

    /* Interrupt */
    enyx_hfp_rx_on_interrupt on_interrupt;
    void * opaque;

    struct enyx_hfp_rx_dev methods;

    struct enyx_hfp_rx_channel channels[];
};

#define dev_to_enyx_hfp_rx_impl(n) ({ \
    struct enyx_hfp_rx_dev * __c = n; \
    container_of(__c, struct enyx_hfp_rx_impl, methods);})

#define channel_to_enyx_hfp_rx_impl(n) ({ \
    struct enyx_hfp_rx_channel * __c = n; \
    container_of(__c, struct enyx_hfp_rx_impl, channels[__c->channel_id]);})

static int
uninstall_dma_buffer_on_channel(struct enyx_hfp_rx_impl * impl, uint16_t channel_id)
{
    int err = 0;

    down(&impl->parent->lock);

    if (enyx_hfp_read8(impl->parent, HFP_RX_CMD_EXEC)) {
        dev_err(&impl->parent->device,
                "Failed to uninstall dma buffer on "
                "channel %u (cmd pending)\n", channel_id);
        err = -EBUSY;
        goto cmd_failed;
    }

    enyx_hfp_write16(impl->parent, HFP_RX_CMD_DEVICE_ID, channel_id);
    enyx_hfp_write64(impl->parent, HFP_RX_CMD_BUFFER_ADDR, 0);
    enyx_hfp_write32(impl->parent, HFP_RX_CMD_BUFFER_SIZE_SHIFT, 0);
    enyx_hfp_write64(impl->parent, HFP_RX_CMD_PTR_ADDR, 0);
    enyx_hfp_write16(impl->parent, HFP_RX_CMD_INTERRUPT_ID, 0);

    enyx_hfp_read8(impl->parent, HFP_RX_CMD_EXEC);
    enyx_hfp_write8(impl->parent, HFP_RX_CMD_EXEC, 1);

    if (enyx_hfp_wait_until_val_is(impl->parent, HFP_RX_CMD_EXEC, 0) < 0) {
        dev_err(&impl->parent->device,
                "Failed to uninstall dma buffer on channel %u\n", channel_id);
        err = -EIO;
        goto cmd_failed;
    }

    dev_dbg(&impl->parent->device, "Uninstalled channel %d dma buffer\n",
            channel_id);

cmd_failed:
    up(&impl->parent->lock);

    return err;
}

static void
free_ctrl_interrupt(struct enyx_hfp_rx_impl * impl,
                    uint16_t channel_id)
{
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];

    free_irq(channel->irq, channel);
    enyx_hfp_irq_list_put(impl->irq_list, channel->irq, channel->int_bus_index);
}

static void
disable_channel_v0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);

    if (channel_id >= impl->channel_count)
        return;

    uninstall_dma_buffer_on_channel(impl, channel_id);

    if (impl->on_interrupt)
        free_ctrl_interrupt(impl, channel_id);
}

static size_t
get_channel_count_v0_0(struct enyx_hfp_rx_dev * dev)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return impl->channel_count;
}

static size_t
get_mtu_v0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    return channel->mtu;
}

static size_t
get_alignment_v0_0(struct enyx_hfp_rx_dev * dev)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return impl->coherency_line_size;
}

static size_t
get_current_buffer_size_shift(struct enyx_hfp_rx_channel * channel)
{
    return (channel->buffer_size_shift) ? channel->buffer_size_shift
                                        : channel->default_buffer_size_shift;
}

static size_t
get_buffers_size_v0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    return 1ULL << get_current_buffer_size_shift(channel);
}

static size_t
get_ctrl_size_v0_0(struct enyx_hfp_rx_dev * dev)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return impl->channel_count * impl->ctrl_increment;
}

static size_t
get_ptrs_size_v0_1(struct enyx_hfp_rx_dev * dev)
{
    return PAGE_SIZE;
}

static size_t
get_info_32_v0_1(struct enyx_hfp_rx_impl* impl,
                 uint16_t channel_id,
                 int register_offset)
{
    size_t count;

    if (down_interruptible(&impl->parent->lock))
        return -1ULL;

    enyx_hfp_write8(impl->parent, HFP_RX_INFO_DEVICE_ID, channel_id);
    udelay(1);
    count = enyx_hfp_read32(impl->parent, register_offset);

    up(&impl->parent->lock);

    return count;
}

static size_t
get_info_32_v0_2(struct enyx_hfp_rx_impl * impl,
                 uint16_t channel_id,
                 int register_offset)
{
    return enyx_hfp_read32_locked_v0_2(impl->parent,
            register_offset,
            HFP_RX_INFO_DEVICE_ID,
            HFP_RX_INFO_PAGE_READY,
            channel_id);
}

static size_t
get_info_16_v0_2(struct enyx_hfp_rx_impl * impl,
                 uint16_t channel_id,
                 int register_offset)
{
    return enyx_hfp_read16_locked_v0_2(impl->parent,
            register_offset,
            HFP_RX_INFO_DEVICE_ID,
            HFP_RX_INFO_PAGE_READY,
            channel_id);
}

static size_t
get_packet_count_v0_1(struct enyx_hfp_rx_dev * dev,
                      uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_1(impl, channel_id, HFP_RX_INFO_PACKET_COUNT);
}

static size_t
get_backpressure_count_v0_1(struct enyx_hfp_rx_dev * dev,
                            uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_1(impl, channel_id, HFP_RX_INFO_BACKPRESSURE_COUNT);
}

static size_t
get_packet_count_v0_2(struct enyx_hfp_rx_dev * dev,
                      uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_2(impl, channel_id, HFP_RX_INFO_PACKET_COUNT);
}

static size_t
get_backpressure_count_v0_2(struct enyx_hfp_rx_dev * dev,
                            uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_2(impl, channel_id, HFP_RX_INFO_BACKPRESSURE_COUNT);
}

static size_t
get_usage_v0_2(struct enyx_hfp_rx_dev * dev,
               uint16_t channel_id,
               char * buffer, size_t buffer_capacity)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    size_t count = 0;

    if (down_interruptible(&impl->parent->lock))
        return 0ULL;

    enyx_hfp_write8(impl->parent, HFP_RX_INFO_DEVICE_ID, channel_id);

    count = enyx_hfp_read_str_unlocked(impl->parent,
                                  HFP_RX_INFO_USAGE_STR,
                                  HFP_RX_INFO_USAGE_STR_INDEX,
                                  HFP_RX_INFO_PAGE_READY,
                                  buffer, buffer_capacity);

    up(&impl->parent->lock);
    return count;
}

static size_t
get_fifo_errors_v0_5(struct enyx_hfp_rx_dev * dev,
                           uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_2(impl, channel_id, HFP_RX_INFO_FIFO_OVERRUN_COUNT);
}

static size_t
get_errors_v0_5(struct enyx_hfp_rx_dev * dev,
                           uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_2(impl, channel_id,
            HFP_RX_DEVICE_PACKET_COUNT_ERRORS);
}

static size_t
get_truncated_count_v1_0(struct enyx_hfp_rx_dev * dev,
                           uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return get_info_32_v0_2(impl, channel_id,
            HFP_RX_DEVICE_PACKET_COUNT_TRUNCATED);
}

static int
install_dma_buffer_on_channel(struct enyx_hfp_rx_impl * impl, uint16_t channel_id,
                              int attr)
{
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    uint64_t ptrs_buffer_addr;
    int err = 0;

    if (down_interruptible(&impl->parent->lock))
        return -ERESTARTSYS;

    impl->methods.disarm_interrupt(&impl->methods, channel_id);
    channel->buffer_offset = 0;
    channel->buffer_used_size = 0;
    channel->buffer_rd_ptr_offset = impl->ctrl_offset
            + impl->ctrl_increment * channel_id
            + HFP_RX_CTRL_RD_PTR;
    impl->methods.update_rd_ptr(&impl->methods, channel_id);

    /* Reset first header of each buffer */
    memset(channel->data_buffer.virt_addr, 0, sizeof(struct enyx_hfp_rx_msg));

    if (attr & HFP_RX_DEV_CHANNEL_ATTR_PTR_FEEDBACK)
        ptrs_buffer_addr = channel->ptrs_buffer.bus_addr;
    else
        ptrs_buffer_addr = 0ULL;

    if (enyx_hfp_read8(impl->parent, HFP_RX_CMD_EXEC)) {
        dev_err(&impl->parent->device,
                "Failed to install dma buffer on "
                "channel %u (cmd pending)\n", channel_id);
        err = -EBUSY;
        goto cmd_failed;
    }

    enyx_hfp_write8(impl->parent, HFP_RX_CMD_DEVICE_ID, channel_id);
    enyx_hfp_write64(impl->parent, HFP_RX_CMD_BUFFER_ADDR,
                channel->data_buffer.bus_addr);
    enyx_hfp_write64(impl->parent, HFP_RX_CMD_PTR_ADDR,
                ptrs_buffer_addr);
    enyx_hfp_write32(impl->parent, HFP_RX_CMD_BUFFER_SIZE_SHIFT,
                fls(channel->data_buffer.size) - 1);
    enyx_hfp_write16(impl->parent, HFP_RX_CMD_INTERRUPT_ID,
                impl->channels[channel_id].int_bus_index);

    enyx_hfp_read8(impl->parent, HFP_RX_CMD_EXEC);
    enyx_hfp_write8(impl->parent, HFP_RX_CMD_EXEC, 1);

    if (enyx_hfp_wait_until_val_is(impl->parent, HFP_RX_CMD_EXEC, 0) < 0) {
        dev_err(&impl->parent->device,
                "Failed to install dma buffer on "
                "channel %u (cmd failed)\n", channel_id);
        err = -EIO;
        goto cmd_failed;
    }

    dev_dbg(&impl->parent->device,
            "Installed channel %d data buffer at %llx and ptrs buffer at %llx\n",
            channel_id, channel->data_buffer.bus_addr, ptrs_buffer_addr);

cmd_failed:
    up(&impl->parent->lock);
    return err;
}

static irqreturn_t
ctrl_on_irq(int irq, void * opaque)
{
    struct enyx_hfp_rx_channel * channel = opaque;
    struct enyx_hfp_rx_impl * impl = channel_to_enyx_hfp_rx_impl(channel);

    dev_dbg(&impl->parent->device,
            "Irq %d associated with channel %hu received\n",
            irq, channel->channel_id);

    impl->on_interrupt(channel->channel_id, impl->opaque);

    return IRQ_HANDLED;
}

static int
request_ctrl_interrupt(struct enyx_hfp_rx_impl * impl,
                       uint16_t channel_id)
{
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    int err;

    dev_dbg(&impl->parent->device,
            "Installing channel %hd interrupt\n", channel_id);

    channel->channel_id = channel_id;
    err = enyx_hfp_irq_list_get(impl->irq_list,
                           &channel->irq,
                           &channel->int_bus_index);
    if (err < 0)
        goto enyx_hfp_irq_list_get_failed;

    err = request_irq(channel->irq, ctrl_on_irq, 0, "enyx_hfp_rx", channel);
    if (err < 0) {
        dev_err(&impl->parent->device,
                "Can't request irq %d\n", channel->irq);
        goto request_irq_failed;
    }

    channel->int_mask_offset = impl->ctrl_offset
            + impl->ctrl_increment * channel_id
            + HFP_RX_CTRL_INTERRUPT_MASK;

    dev_dbg(&impl->parent->device,
            "Channel %d uses bus interrupt %d mapped to cpu irq %d\n",
            channel_id, channel->int_bus_index, channel->irq);

    return err;

request_irq_failed:
    enyx_hfp_irq_list_put(impl->irq_list, channel->irq, channel->int_bus_index);
enyx_hfp_irq_list_get_failed:
    return err;
}

static int
enable_channel_v0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id, int attr)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    int err;

    if (channel_id >= impl->channel_count) {
        err = -EINVAL;
        goto channel_id_check_failed;
    }

    if (impl->on_interrupt) {
        err = request_ctrl_interrupt(impl, channel_id);
        if (err < 0)
            goto request_ctrl_interrupt_failed;
    }

    err = install_dma_buffer_on_channel(impl, channel_id, attr);
    if (err < 0)
        goto install_dma_buffer_on_channel_failed;

    return 0;

install_dma_buffer_on_channel_failed:
    if (impl->on_interrupt)
        free_ctrl_interrupt(impl, channel_id);
request_ctrl_interrupt_failed:
channel_id_check_failed:
    return err;
}

int
arm_interrupt_v0_0(struct enyx_hfp_rx_dev * dev,
                   uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);

    if (! impl->on_interrupt)
        return -EINVAL;

    if (channel_id >= impl->channel_count)
        return -EINVAL;

    enyx_hfp_write32(impl->parent, impl->channels[channel_id].int_mask_offset, 1);

    return 0;
}

int
disarm_interrupt_v0_0(struct enyx_hfp_rx_dev * dev,
                   uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);

    if (channel_id >= impl->channel_count)
        return -EINVAL;

    enyx_hfp_write32(impl->parent, impl->channels[channel_id].int_mask_offset, 0);

    return 0;
}

#define to_buffer_ptr(c) \
    (const void *)(c->data_buffer.virt_addr + c->buffer_offset)

static int
poll_one_v0_0(struct enyx_hfp_rx_dev * dev,
              uint16_t channel_id,
              enyx_hfp_rx_on_data on_data,
              void * opaque)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * c = &impl->channels[channel_id];
    const struct enyx_hfp_rx_msg * m = to_buffer_ptr(c);
    uint32_t msg_size;

    if (! m->data_size)
        return -EAGAIN;

    on_data(m->data, m->data_size, opaque);

    msg_size = ALIGN(sizeof(*m) + m->data_size, impl->coherency_line_size);
    c->buffer_offset += msg_size;

    /* Rollover if there is less than mtu space remaining
       to the end of the buffer */
    if (c->buffer_offset + sizeof(*m) + c->mtu > c->data_buffer.size)
        c->buffer_offset = 0;

    c->buffer_used_size += msg_size;
    if (c->buffer_used_size > c->data_buffer.size / 2)
        impl->methods.update_rd_ptr(dev, channel_id);

    return 0;
}

static int
update_rd_ptr_v0_0(struct enyx_hfp_rx_dev * dev,
                   uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * c = &impl->channels[channel_id];

    enyx_hfp_write32(impl->parent, c->buffer_rd_ptr_offset, c->buffer_offset);
    c->buffer_used_size = 0;

    dev_dbg(&impl->parent->device,
            "Updated channel %hu read ptr to %u\n",
            channel_id, c->buffer_offset);

    return 0;
}

static int
mmap_ctrl_v0_0(struct enyx_hfp_rx_dev * dev, struct vm_area_struct *vma,
               uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    unsigned long io_offset = vma->vm_pgoff << PAGE_SHIFT,
                  base_addr = impl->parent->bus->phys_addr +
                              impl->parent->bus_offset +
                              impl->ctrl_offset +
                              channel_id * impl->ctrl_increment,
                  io_start = base_addr + io_offset,
                  vm_size = vma->vm_end - vma->vm_start;
    int err;

    dev_dbg(&impl->parent->device,
            "Requested ctrl mapping of %lu bytes at %lx\n", vm_size, io_start);

    if (vm_size > PAGE_ALIGN(impl->ctrl_increment - io_offset)) {
        dev_err(&impl->parent->device,
                "Could not grant ctrl mapping of %lu bytes as it is bigger than ctrl increment (%lu) - offset (%lu)\n",
                vm_size, impl->ctrl_increment, io_offset);
        err = -EINVAL;
        goto err_invalid_arg;
    }

    err = io_remap_pfn_range(vma,
                             vma->vm_start,
                             io_start >> PAGE_SHIFT,
                             vm_size,
                             pgprot_noncached(vma->vm_page_prot));

err_invalid_arg:
    return err;
}

static int
mmap_buffer(struct enyx_hfp_rx_impl * impl, struct vm_area_struct *vma,
            struct enyx_hfp_rx_coherent_buffer * buffer)
{
    unsigned long vm_size = vma->vm_end - vma->vm_start,
                  vm_offset = vma->vm_pgoff << PAGE_SHIFT;
    void * virt_addr = buffer->virt_addr + vm_offset;
    int err;

    dev_dbg(&impl->parent->device,
            "Requested buffer mapping of %zu bytes\n", buffer->size);

    if (vm_size > buffer->size - vm_offset) {
        dev_err(&impl->parent->device,
                "Could not grant buffer mapping of %lu bytes as it is bigger than buffer size (%lu) - offset (%lu)\n",
                vm_size, buffer->size, vm_offset);
        err = -EINVAL;
        goto err_invalid_arg;
    }

    err = remap_pfn_range(vma,
                          vma->vm_start,
                          virt_to_phys(virt_addr) >> PAGE_SHIFT,
                          vm_size,
                          vma->vm_page_prot);

err_invalid_arg:
    return err;
}

static int
mmap_ptrs_v0_1(struct enyx_hfp_rx_dev * dev, struct vm_area_struct *vma,
               uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return mmap_buffer(impl, vma, &impl->channels[channel_id].ptrs_buffer);
}

static int
mmap_data_v0_0(struct enyx_hfp_rx_dev * dev, struct vm_area_struct *vma,
               uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    return mmap_buffer(impl, vma, &impl->channels[channel_id].data_buffer);
}

static void
destroy_dma_buffer(struct enyx_hfp_rx_impl * impl,
                   struct enyx_hfp_rx_coherent_buffer * buffer)
{
    dma_free_coherent(impl->dma,
                      buffer->size,
                      buffer->virt_addr,
                      buffer->bus_addr);
    buffer->virt_addr = NULL;
}

static void
destroy_dma_buffers(struct enyx_hfp_rx_impl * impl)
{
    uint32_t i;

    for (i = 0; i != impl->channel_count; ++i)
    {
        destroy_dma_buffer(impl, &impl->channels[i].data_buffer);
        destroy_dma_buffer(impl, &impl->channels[i].ptrs_buffer);
    }
}

void
enyx_hfp_rx_dev_destroy(struct enyx_hfp_rx_dev * dev)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);

    destroy_dma_buffers(impl);

    put_device(&impl->parent->device);

    kfree(impl);
}
EXPORT_SYMBOL(enyx_hfp_rx_dev_destroy);

static int
setup_cache_line(struct enyx_hfp_rx_impl * impl, size_t coherency_line_size)
{
    const size_t min_size = 1ULL << enyx_hfp_read16(impl->parent,
            HFP_RX_CACHE_LINE_SHIFT_MIN);
    const size_t max_size = 1ULL << enyx_hfp_read16(impl->parent,
            HFP_RX_CACHE_LINE_SHIFT_MAX);

    dev_dbg(&impl->parent->device, "Reported min CL size is %zu\n", min_size);
    dev_dbg(&impl->parent->device, "Reported max CL size is %zu\n", max_size);

    coherency_line_size = max(coherency_line_size, min_size);

    if (max_size < coherency_line_size) {
        dev_err(&impl->parent->device,
                "Reported max CL size (%zu) is smaller"
                " than CPU CL size (%zu)\n",
                max_size, coherency_line_size);
        goto cache_size_check_failed;
    }

    enyx_hfp_write8(impl->parent, HFP_RX_CACHE_LINE_SHIFT,
               fls(coherency_line_size) - 1);

    impl->coherency_line_size = coherency_line_size;

    return 0;

cache_size_check_failed:
    return -1;
}

static int
alloc_dma_buffer(struct enyx_hfp_rx_coherent_buffer * buffer,
                 struct device * dma,
                 size_t buffer_size)
{
    buffer->size = buffer_size;
    buffer->virt_addr = dma_alloc_coherent(dma,
                                        buffer->size,
                                        &buffer->bus_addr,
                                        GFP_KERNEL);
    if (! buffer->virt_addr) {
        dev_err(dma,
                "Can't allocate %zu bytes DMA buffer\n", buffer->size);
        goto dma_alloc_attrs_failed;
    }

    dev_dbg(dma, "Allocated a %zu bytes DMA buffer\n", buffer->size);

    return 0;

dma_alloc_attrs_failed:
    return -1;
}

static int
alloc_dma_buffers(struct enyx_hfp_rx_impl * impl,
                  size_t buffers_size_shift)
{
    uint32_t i;
    const size_t max_size_shift = enyx_hfp_read16(impl->parent,
                                            HFP_RX_BUFFER_SIZE_SHIFT_MAX);

    if (max_size_shift < buffers_size_shift) {
        dev_err(&impl->parent->device,
                "Reported max DMA buffer size shift (%zu)"
                " is smaller than requested size shift (%zu)\n",
                max_size_shift,
                buffers_size_shift);
        goto buffer_size_check_failed;
    }

    impl->dma = &impl->parent->device;

    while (impl->dma && ! impl->dma->dma_mask)
            impl->dma = impl->dma->parent;

    if (! impl->dma) {
        dev_err(&impl->parent->device,
                "Can't find a DMA capable d "
                "in this d hierairchy\n");
        goto is_device_dma_capable_failed;
    }

    impl->buffers_size_shift = buffers_size_shift;

    for (i = 0; i != impl->channel_count; ++i)
        impl->channels[i].default_buffer_size_shift = buffers_size_shift;

    for (i = 0; i != impl->channel_count; ++i)
        if (alloc_dma_buffer(&impl->channels[i].ptrs_buffer,
                             impl->dma, PAGE_SIZE) < 0)
            goto alloc_dma_ptrs_buffer_failed;

    return 0;

alloc_dma_ptrs_buffer_failed:
    while (i)
        destroy_dma_buffer(impl, &impl->channels[--i].ptrs_buffer);
is_device_dma_capable_failed:
buffer_size_check_failed:
    return -EIO;
}

static int
reset(struct enyx_hfp_rx_impl * impl)
{
    enyx_hfp_write8(impl->parent, HFP_RX_RESET, 1);

    if (enyx_hfp_wait_until_val_is(impl->parent, HFP_RX_RESET, 0) < 0) {
        dev_err(&impl->parent->device, "Failed to reset device\n");
        goto reset_cmd_failed;
    }

    return 0;

reset_cmd_failed:
    return -1;
}

static int
alloc_dma_data_buffer_0_0(struct enyx_hfp_rx_dev *dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    if (channel->data_buffer.virt_addr)
        return -EBUSY;
    if (alloc_dma_buffer(&channel->data_buffer,
                         impl->dma,
                         1ULL << get_current_buffer_size_shift(channel)) < 0)
        return -EIO;
    return 0;
}

static void
free_dma_data_buffer_0_0(struct enyx_hfp_rx_dev *dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    if (! channel->data_buffer.virt_addr)
        return;
    destroy_dma_buffer(impl, &channel->data_buffer);
}

static int
set_buffer_width_0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id,
                     size_t new_buffer_width)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    const size_t max_size_shift = enyx_hfp_read16(impl->parent,
                                            HFP_RX_BUFFER_SIZE_SHIFT_MAX);

    if (channel->data_buffer.virt_addr) {
        dev_err(&impl->parent->device,
                "Attempted to set buffer width of open channel %u\n",
                channel_id);
        return -EBUSY;
    }

    if (max_size_shift < new_buffer_width) {
        dev_err(&impl->parent->device,
                "Reported max DMA buffer size shift (%zu)"
                " is smaller than requested size shift (%zu)\n",
                max_size_shift,
                new_buffer_width);
        return -EIO;
    }
    channel->buffer_size_shift = new_buffer_width;
    return 0;
}

static size_t
get_buffer_width_0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];

    return get_current_buffer_size_shift(channel);
}

static int
set_default_buffer_width_0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id,
                             size_t new_buffer_width)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];
    const size_t max_size_shift = enyx_hfp_read16(impl->parent,
                                            HFP_RX_BUFFER_SIZE_SHIFT_MAX);

    if (new_buffer_width == 0) {
        dev_err(&impl->parent->device,
                "Default size shift (%zu) is too small\n",
                new_buffer_width);
        return -EIO;
    }
    if (max_size_shift < new_buffer_width) {
        dev_err(&impl->parent->device,
                "Reported max DMA buffer size shift (%zu)"
                " is smaller than requested default size shift (%zu)\n",
                max_size_shift,
                new_buffer_width);
        return -EIO;
    }
    channel->default_buffer_size_shift = new_buffer_width;
    return 0;
}

static size_t
get_default_buffer_width_0_0(struct enyx_hfp_rx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_rx_impl * impl = dev_to_enyx_hfp_rx_impl(dev);
    struct enyx_hfp_rx_channel * channel = &impl->channels[channel_id];

    return channel->default_buffer_size_shift;
}

static void
bind_methods(struct enyx_hfp_rx_impl * impl)
{
    impl->methods.get_mtu = get_mtu_v0_0;
    impl->methods.get_alignment = get_alignment_v0_0;
    impl->methods.get_buffers_size = get_buffers_size_v0_0;
    impl->methods.get_ctrl_size = get_ctrl_size_v0_0;
    impl->methods.get_channel_count = get_channel_count_v0_0;
    impl->methods.enable_channel = enable_channel_v0_0;
    impl->methods.disable_channel = disable_channel_v0_0;
    impl->methods.poll_one = poll_one_v0_0;
    impl->methods.arm_interrupt = arm_interrupt_v0_0;
    impl->methods.disarm_interrupt = disarm_interrupt_v0_0;
    impl->methods.update_rd_ptr = update_rd_ptr_v0_0;
    impl->methods.mmap_ctrl = mmap_ctrl_v0_0;
    impl->methods.mmap_data = mmap_data_v0_0;
    impl->methods.alloc_dma_data_buffer = alloc_dma_data_buffer_0_0;
    impl->methods.free_dma_data_buffer = free_dma_data_buffer_0_0;
    impl->methods.set_buffer_width = set_buffer_width_0_0;
    impl->methods.get_buffer_width = get_buffer_width_0_0;
    impl->methods.set_default_buffer_width = set_default_buffer_width_0_0;
    impl->methods.get_default_buffer_width = get_default_buffer_width_0_0;

    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(0, 1)) {
        impl->methods.mmap_ptrs = mmap_ptrs_v0_1;
        impl->methods.get_ptrs_size = get_ptrs_size_v0_1;
        impl->methods.get_packet_count = get_packet_count_v0_1;
        impl->methods.get_backpressure_count = get_backpressure_count_v0_1;
    }

    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(0, 2)) {
        impl->methods.get_packet_count = get_packet_count_v0_2;
        impl->methods.get_backpressure_count = get_backpressure_count_v0_2;
        impl->methods.get_usage = get_usage_v0_2;
    }

    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(0, 5)) {
        impl->methods.get_fifo_errors = get_fifo_errors_v0_5;
        impl->methods.get_errors = get_errors_v0_5;
    }

    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(1, 0)) {
        impl->methods.get_truncated_count = get_truncated_count_v1_0;
    }
}

static void init_channels_v0_0(struct enyx_hfp_rx_impl * impl)
{
    uint16_t i;
    for (i = 0; i < impl->channel_count; i++) {
        struct enyx_hfp_rx_channel * channel = &impl->channels[i];
        channel->mtu = impl->global_mtu;
    }
}

static void init_channels_v1_0(struct enyx_hfp_rx_impl *impl)
{
    uint16_t i;
    for (i = 0; i < impl->channel_count; i++) {
        struct enyx_hfp_rx_channel * channel = &impl->channels[i];
        channel->mtu = get_info_16_v0_2(impl, i,
                HFP_RX_DEVICE_MTU);
    }
}

struct enyx_hfp_rx_dev *
enyx_hfp_rx_dev_create(struct enyx_hfp_device * parent,
                  enyx_hfp_rx_on_interrupt on_interrupt,
                  void * opaque,
                  size_t buffers_size_shift,
                  size_t coherency_line_size)
{
    uint32_t channel_count;
    struct enyx_hfp_rx_impl * impl;

    if (! get_device(&parent->device))
        goto get_device_failed;

    if (parent->size < 16 * 4) {
        dev_err(&parent->device,
                "Reported device memory range smaller than expected\n");
        goto size_check_failed;
    }

    if (parent->id.version_major > 1) {
        dev_err(&parent->device,
                "Reported device major version %hhu is incompatible\n",
                parent->id.version_major);
        goto major_check_failed;
    }

    channel_count = enyx_hfp_read8(parent, HFP_RX_DEVICE_COUNT);

    impl = kzalloc(sizeof(*impl) + sizeof(*impl->channels) * channel_count,
                   GFP_KERNEL);
    if (! impl) {
        dev_err(&parent->device, "Can't allocate rx\n");
        goto kzalloc_failed;
    }

    impl->parent = parent;
    impl->on_interrupt = on_interrupt;
    impl->opaque = opaque;
    impl->irq_list = parent->bus->irq_list;
    impl->channel_count = channel_count;
    impl->global_mtu = enyx_hfp_read16(parent, HFP_RX_MTU);

    impl->ctrl_offset = enyx_hfp_read32(parent, HFP_RX_CTRL_BASE_OFFSET_DW) * 4;
    impl->ctrl_increment = enyx_hfp_read32(parent, HFP_RX_CTRL_INCREMENT_DW) * 4;

    dev_dbg(&parent->device,
            "Ctrl(s) located at offset 0x%zx every %zu bytes\n",
            impl->ctrl_offset, impl->ctrl_increment);

    dev_dbg(&parent->device, "Found %u channels\n", impl->channel_count);

    bind_methods(impl);

    init_channels_v0_0(impl);
    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(1, 0)) {
        init_channels_v1_0(impl);
    }

    if (reset(impl))
        goto reset_failed;

    if (setup_cache_line(impl, coherency_line_size) < 0)
        goto setup_cache_line_failed;

    /* Setup contigous DMA buffers ASAP to ensure
     * memory fragmentation won't be an issue while
     * device will be open later */
    if (alloc_dma_buffers(impl, buffers_size_shift) < 0)
        goto alloc_dma_buffer_failed;

    return &impl->methods;

alloc_dma_buffer_failed:
setup_cache_line_failed:
reset_failed:
    kfree(impl);
kzalloc_failed:
major_check_failed:
size_check_failed:
    put_device(&parent->device);
get_device_failed:
    return NULL;
}
EXPORT_SYMBOL(enyx_hfp_rx_dev_create);

