/*
 * HFP net implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_net_dev.h"

#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/fs.h>
#include <linux/if_vlan.h>
#include <linux/ip.h>
#include <linux/mm.h>
#include <linux/netdevice.h>
#include <linux/slab.h>
#include <linux/timer.h>
#include <linux/version.h>
#include <linux/workqueue.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_rx_dev.h"
#include "enyx_hfp_tx_dev.h"


#define HFP_NET_RX_BUFFER_SIZE_SHIFT 16

/* A 8-bit r/o register: interfaces count */
#define HFP_NET_DEVICE_COUNT                 HFP_REG_ADDR(4, 0)

/* A 8-bit r/o register: interface id */
#define HFP_NET_DEVICE_INFO_PAGE_ID          HFP_REG_ADDR(5, 0)

/* A 16-bit r/o register: usage str index */
#define HFP_NET_DEVICE_INFO_USAGE_STR_INDEX  HFP_REG_ADDR(5, 2)

/* A 8-bit r/o register: interface page valid */
#define HFP_NET_DEVICE_INFO_PAGE_READY       HFP_REG_ADDR(6, 0)

/* A 48-bit r/o register: MAC address */
#define HFP_NET_DEVICE_INFO_MAC_ADDRESS      HFP_REG_ADDR(7, 0)

/* A 32-bit r/o register: usage str */
#define HFP_NET_DEVICE_INFO_USAGE_STR        HFP_REG_ADDR(9, 0)

/* A 1-bit r/o register: carrier status */
#define HFP_NET_DEVICE_INFO_STATUS           HFP_REG_ADDR(10, 0)

/* A 32-bit r/o register: filter total input frame count */
#define HFP_NET_DEVICE_INFO_INPUT_COUNT      HFP_REG_ADDR(11, 0)
/* A 32-bit r/o register: filter dropped frame count */
#define HFP_NET_DEVICE_INFO_DROP_COUNT       HFP_REG_ADDR(12, 0)

/* A 16-bit r/o register: device whitelist page offset in global whitelist */
#define HFP_NET_DEVICE_WHITELIST_PAGE_OFFSET HFP_REG_ADDR(20, 0)
/* A 32-bit r/o register: device whitelist size */
#define HFP_NET_DEVICE_WHITELIST_ENTRY_COUNT HFP_REG_ADDR(20, 2)

/* A 1-bit r/w register: filter enable */
#define HFP_NET_DEVICE_FILTER_ENABLE         HFP_REG_ADDR(21, 0)

/* A 1-bit r/o register: enable unicast promicuous mode */
#define HFP_NET_DEVICE_UCAST_PROMISC_ENABLE  HFP_REG_ADDR(21, 1)
/* A 1-bit r/o register: enable multicast promicuous mode */
#define HFP_NET_DEVICE_MCAST_PROMISC_ENABLE  HFP_REG_ADDR(21, 2)

/* A 8-bit r/o register: whitelist page valid */
#define HFP_NET_DEVICE_WHITELIST_PAGE_READY    HFP_REG_ADDR(30, 0)
/* A 8-bit w/o register: whitelist page id */
#define HFP_NET_DEVICE_WHITELIST_PAGE_ID       HFP_REG_ADDR(31, 0)
/* A 1-bit r/w register: enable whitelist entry */
#define HFP_NET_DEVICE_WHITELIST_ENTRY_ENABLE  HFP_REG_ADDR(33, 0)
/* A 48-bit r/w register: whitelist entry address */
#define HFP_NET_DEVICE_WHITELIST_ENTRY_ADDR    HFP_REG_ADDR(34, 0)

#define HFP_NET_DEVICE_WHITELIST_ENTRY_ADDR_SIZE 48


struct enyx_hfp_net_dev {
    struct enyx_hfp_device * parent;

    struct enyx_hfp_tx_dev * tx_dev;
    struct enyx_hfp_rx_dev * rx_dev;

    uint32_t coherency_line_size;
    uint16_t channel_count;

    bool is_igmp_only;

    size_t
    (*get_usage)(struct enyx_hfp_net_dev * dev, uint16_t channel_id,
                 char * buffer, size_t buffer_capacity);
    bool
    (*get_carrier)(struct enyx_hfp_net_dev * dev, uint16_t channel_id);

    void
    (*enable_filtering)(struct enyx_hfp_net_dev *dev, uint16_t channel_id);

    struct net_device * net_devs[0];
};

struct enyx_hfp_net_dev_priv {
    bool is_open;
    struct enyx_hfp_net_dev * dev;
    uint16_t id;
    uint16_t port_id;
    uint32_t max_mtu;

    struct rtnl_link_stats64 stats;

    struct timer_list tx_timer;
    struct sk_buff * pending_tx_skb;
    uint16_t tx_timeout_count;

    struct napi_struct napi;
    struct delayed_work watchdog_task;

    uint16_t whitelist_page_offset;
    uint32_t whitelist_entry_count;

    bool ucast_promisc;
    bool mcast_promisc;
};

#define napi_to_net_dev_priv(n) \
    container_of(n, struct enyx_hfp_net_dev_priv, napi)

#define priv_to_net_dev(p) \
    p->dev->net_devs[p->id]

/* information reporting for ethtool */
#define rx_get_attr(attr) \
    static size_t rx_get_##attr(struct enyx_hfp_net_dev_priv *priv) \
    { \
        struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev; \
        if (rx_dev->get_##attr) \
            return rx_dev->get_##attr(rx_dev, priv->id); \
        return -1ULL; \
    }

rx_get_attr(backpressure_count);
rx_get_attr(packet_count);
rx_get_attr(fifo_errors);
rx_get_attr(truncated_count);

static size_t rx_get_crc_errors(struct enyx_hfp_net_dev_priv *priv)
{
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;
    if (rx_dev->get_errors)
        return rx_dev->get_errors(rx_dev, priv->id);

    return -1ULL;
}

static size_t rx_netdev_get_errors(struct enyx_hfp_net_dev_priv *priv)
{
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;
    size_t errors = priv->stats.rx_errors;

    if (rx_dev->get_fifo_errors)
        errors += rx_dev->get_fifo_errors(rx_dev, priv->id);

    if (rx_dev->get_errors)
        errors += rx_dev->get_errors(rx_dev, priv->id);

    return errors;
}

static int
select_whitelist_locked(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                        uint16_t whitelist_id)
{
    uint16_t page_offset;
    struct enyx_hfp_net_dev_priv *priv = netdev_priv(dev->net_devs[channel_id]);

    page_offset = priv->whitelist_page_offset + whitelist_id;

    enyx_hfp_write16(dev->parent,
                     HFP_NET_DEVICE_WHITELIST_PAGE_ID, page_offset);
    if (enyx_hfp_wait_until_val_is(
                    dev->parent, HFP_NET_DEVICE_WHITELIST_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select whitelist page\n");
        goto whitelist_page_select_failed;
    }

    return 0;

whitelist_page_select_failed:
    return -1;
}

static void
set_device_byte(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                uint16_t reg, uint8_t value)
{
    if (down_interruptible(&dev->parent->lock))
        return;

    enyx_hfp_write32(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, channel_id);
    if (enyx_hfp_wait_until_val_is(dev->parent,
                                   HFP_NET_DEVICE_INFO_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select interface page\n");
        goto interface_page_select_failed;
    }

    enyx_hfp_write8(dev->parent, reg, value);

interface_page_select_failed:
    up(&dev->parent->lock);
}

static void
enable_filtering(struct enyx_hfp_net_dev *dev, uint16_t channel_id)
{
    set_device_byte(dev, channel_id, HFP_NET_DEVICE_FILTER_ENABLE, 1);
    netdev_dbg(dev->net_devs[channel_id], "filtering feature enabled");
}

static void
set_ucast_promisc(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                  bool enable)
{
    struct enyx_hfp_net_dev_priv *priv = netdev_priv(dev->net_devs[channel_id]);

    set_device_byte(dev, channel_id,
                    HFP_NET_DEVICE_UCAST_PROMISC_ENABLE, enable);
    priv->ucast_promisc = enable;
}

static void
set_mcast_promisc(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                  bool enable)
{
    struct enyx_hfp_net_dev_priv *priv = netdev_priv(dev->net_devs[channel_id]);

    set_device_byte(dev, channel_id,
                    HFP_NET_DEVICE_MCAST_PROMISC_ENABLE, enable);
    priv->mcast_promisc = enable;

}

static size_t
get_whitelist_count(struct enyx_hfp_net_dev * dev, uint16_t channel_id)
{
    uint8_t result = 0;
    uint16_t page_offset = 0;

    if (down_interruptible(&dev->parent->lock))
        goto interface_page_select_lock_failed;

    enyx_hfp_write16(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, channel_id);
    if (enyx_hfp_wait_until_val_is(dev->parent,
                                   HFP_NET_DEVICE_INFO_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select interface page\n");
        goto interface_page_select_failed;
    }

    result = enyx_hfp_read16(dev->parent, HFP_NET_DEVICE_WHITELIST_ENTRY_COUNT);

    page_offset = enyx_hfp_read16(dev->parent,
                                  HFP_NET_DEVICE_WHITELIST_PAGE_OFFSET);

interface_page_select_failed:
    up(&dev->parent->lock);
interface_page_select_lock_failed:
    return result;
}

static void
set_whitelist_entry(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                    uint16_t whitelist_id, unsigned char *addr, size_t addr_len)
{
    uint64_t addr_reg = 0;
    uint8_t i;

    bool is_len_invalid = addr_len > HFP_NET_DEVICE_WHITELIST_ENTRY_ADDR_SIZE;
    WARN_ON_ONCE(is_len_invalid);
    if (is_len_invalid)
        addr_len = HFP_NET_DEVICE_WHITELIST_ENTRY_ADDR_SIZE;

    for (i = 0; i != addr_len; ++i)
        addr_reg = (addr_reg << 8) | addr[i];

    if (down_interruptible(&dev->parent->lock))
        return;

    if (select_whitelist_locked(dev, channel_id, whitelist_id) < 0)
        goto select_whitelist_failed;


    enyx_hfp_write64(dev->parent, HFP_NET_DEVICE_WHITELIST_ENTRY_ADDR,
                     addr_reg);
    enyx_hfp_write8(dev->parent, HFP_NET_DEVICE_WHITELIST_ENTRY_ENABLE, 1);

    netdev_dbg(dev->net_devs[channel_id],
               "added address to filter whitelist: %012llx", addr_reg);

select_whitelist_failed:
    up(&dev->parent->lock);
}

static void
disable_whitelist_entry(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                        uint16_t whitelist_id)
{
    if (down_interruptible(&dev->parent->lock))
        return;

    if (select_whitelist_locked(dev, channel_id, whitelist_id) < 0)
        goto select_whitelist_failed;

    enyx_hfp_write8(dev->parent, HFP_NET_DEVICE_WHITELIST_ENTRY_ENABLE, 0);

select_whitelist_failed:
    up(&dev->parent->lock);
}

static size_t
dump_filter_whitelist(struct enyx_hfp_net_dev_priv *priv)
{
    struct enyx_hfp_net_dev *dev = priv->dev;
    struct net_device *netdev = priv_to_net_dev(priv);

    uint16_t i;
    uint16_t whl_count;
    uint16_t enabled_count = 0;

    whl_count = priv->whitelist_entry_count;
    if (down_interruptible(&dev->parent->lock))
        return 0;

    for (i = 0; i < whl_count; ++i) {
        uint64_t addr;
        uint8_t enable;

        if (select_whitelist_locked(dev, priv->id, i) < 0)
            break;

        addr = enyx_hfp_read64(dev->parent,
                               HFP_NET_DEVICE_WHITELIST_ENTRY_ADDR);
        enable = enyx_hfp_read8(dev->parent,
                                HFP_NET_DEVICE_WHITELIST_ENTRY_ENABLE);

        if (enable & 1) {
            netdev_dbg(netdev,
                       "filter whitelist entry %d enabled, addr: %012llx",
                       i, addr);
            enabled_count += 1;
        }
        else
            netdev_dbg(netdev, "filter whitelist entry %d disabled", i);
    }

    up(&dev->parent->lock);

    return enabled_count;
}


static size_t
enyx_hfp_rx_filter_get_stat(struct enyx_hfp_net_dev_priv *priv, uint32_t stat_reg)
{
    struct enyx_hfp_net_dev *dev = priv->dev;
    size_t count = 0;

    if (down_interruptible(&dev->parent->lock))
        return 0;

    enyx_hfp_write16(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, priv->id);
    if (enyx_hfp_wait_until_val_is(dev->parent,
                                   HFP_NET_DEVICE_INFO_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select interface page\n");
        goto interface_page_select_failed;
    }

    count = enyx_hfp_read32(dev->parent, stat_reg);

interface_page_select_failed:
    up(&dev->parent->lock);

    return count;
}

static size_t
rx_filter_get_input_count(struct enyx_hfp_net_dev_priv *priv)
{
    return enyx_hfp_rx_filter_get_stat(priv, HFP_NET_DEVICE_INFO_INPUT_COUNT);
}

static size_t
rx_filter_get_drop_count(struct enyx_hfp_net_dev_priv *priv)
{
    return enyx_hfp_rx_filter_get_stat(priv, HFP_NET_DEVICE_INFO_DROP_COUNT);
}

static size_t
rx_filter_get_unicast_promisc_en(struct enyx_hfp_net_dev_priv *priv)
{
    return priv->ucast_promisc;
}

static size_t
rx_filter_get_multicast_promisc_en(struct enyx_hfp_net_dev_priv *priv)
{
    return priv->mcast_promisc;
}

static size_t
rx_filter_get_whitelist_capacity(struct enyx_hfp_net_dev_priv *priv)
{
    return priv->whitelist_entry_count;
}

static size_t
rx_filter_get_whitelist_usage(struct enyx_hfp_net_dev_priv *priv)
{
    return dump_filter_whitelist(priv);
}

static void
on_rx_interrupt(uint16_t channel_id, void * opaque)
{
    struct enyx_hfp_net_dev * dev = opaque;
    struct net_device * net_dev = dev->net_devs[channel_id];
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);

    /* Restart polling on rx */
    napi_schedule(&priv->napi);
}

static int
arm_rx_interrupt(struct enyx_hfp_net_dev_priv * priv)
{
    uint16_t channel_id = priv->id;
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;

    int ret = rx_dev->arm_interrupt(rx_dev, channel_id);

    if (ret < 0)
        netdev_err(priv_to_net_dev(priv),
                   "Failed to arm rx interrupt on interface %hu\n",
                   channel_id);
    return ret;
}

static int
open_interface(struct net_device * net_dev)
{
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;
    struct enyx_hfp_tx_dev * tx_dev = priv->dev->tx_dev;
    int err;

    if (! enyx_hfp_device_get(priv->dev->parent)) {
        err = -ENODEV;
        goto enyx_hfp_device_get_failed;
    }

    netdev_dbg(net_dev, "Open\n");

    err = rx_dev->enable_channel(rx_dev, priv->id, 0);
    if (err < 0)
        goto enyx_hfp_rx_dev_enable_channel_failed;

    err = tx_dev->enable_channel(tx_dev, priv->id);
    if (err < 0)
        goto enyx_hfp_tx_dev_enable_channel_failed;

    napi_enable(&priv->napi);

    err = arm_rx_interrupt(priv);
    if (err < 0)
        goto arm_interrupt_failed;

    priv->is_open = true;

    return 0;

arm_interrupt_failed:
    napi_disable(&priv->napi);
    tx_dev->disable_channel(tx_dev, priv->id);
enyx_hfp_tx_dev_enable_channel_failed:
    rx_dev->disable_channel(rx_dev, priv->id);
enyx_hfp_rx_dev_enable_channel_failed:
    enyx_hfp_device_put(priv->dev->parent);
enyx_hfp_device_get_failed:
    return err;
}

static int
close_interface(struct net_device * net_dev)
{
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;
    struct enyx_hfp_tx_dev * tx_dev = priv->dev->tx_dev;

    netdev_dbg(net_dev, "Close\n");

    if (! priv->is_open)
        return 0;

    del_timer_sync(&priv->tx_timer);
    if (priv->pending_tx_skb) {
        kfree_skb(priv->pending_tx_skb);
        priv->pending_tx_skb = NULL;
    }

    napi_disable(&priv->napi);

    tx_dev->disable_channel(tx_dev, priv->id);

    rx_dev->disable_channel(rx_dev, priv->id);

    priv->is_open = false;

    enyx_hfp_device_put(priv->dev->parent);

    return 0;
}

#define HFP_NET_TX_RETRIES 10
#define HFP_NET_TX_TIMEOUT (HZ / HFP_NET_TX_RETRIES)

static void
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 14, 0)
on_tx_buffer_full(struct timer_list *t)
{
    struct enyx_hfp_net_dev_priv * priv = from_timer(priv, t, tx_timer);
#else
on_tx_buffer_full(unsigned long opaque)
{
    struct enyx_hfp_net_dev_priv * priv = (void *)opaque;
#endif
    struct enyx_hfp_tx_dev * tx_dev = priv->dev->tx_dev;
    struct net_device * net_dev = priv_to_net_dev(priv);
    int err;

    /* The tx timer may fire while the interface is closing.
       The close function will perform the cleanup. */
    if (! netif_running(net_dev))
        return;

    netdev_dbg(net_dev, "Xmit (after buffer full)\n");

    err = tx_dev->send(tx_dev, priv->id,
                       priv->pending_tx_skb->data,
                       priv->pending_tx_skb->len);
    if (err == -EAGAIN && ++ priv->tx_timeout_count < HFP_NET_TX_RETRIES) {
        mod_timer(&priv->tx_timer, jiffies + HFP_NET_TX_TIMEOUT);
    }
    else {
        if (err < 0) {
            netdev_dbg(net_dev, "Failed to send skb\n");
            ++ priv->stats.tx_errors;
            kfree_skb(priv->pending_tx_skb);
        }
        else {
            priv->stats.tx_bytes += priv->pending_tx_skb->len;
            ++ priv->stats.tx_packets;
            consume_skb(priv->pending_tx_skb);
        }

        priv->pending_tx_skb = NULL;
        netif_wake_queue(net_dev);
    }
}

static bool
is_filtered(const uint8_t * data, uint32_t data_size)
{
    const struct ethhdr * eh = (const void *)data;
    int offset = sizeof(*eh);
    __be16 proto;

    if (sizeof(*eh) > data_size) {
        pr_debug("Ethernet overrun detected\n");
        goto buffer_overrun_check_failed;
    }

    proto = ntohs(eh->h_proto);

    /* Strip vlan 802.1Q header(s) */
    while (proto == ETH_P_8021Q || proto == ETH_P_8021AD) {
        const struct vlan_hdr * vh = (const void *)(data + offset);

        if (offset + sizeof(*vh) > data_size) {
            pr_debug("Vlan overrun detected\n");
            goto buffer_overrun_check_failed;
        }

        proto = ntohs(vh->h_vlan_encapsulated_proto);
        offset += sizeof(*vh);
    }

#if 0
    if (proto == htons(ETH_P_ARP))
        return false;
#endif

    if (proto == ETH_P_IP)
    {
        const struct iphdr * ih = (const void *)(data + offset);

        if (offset + sizeof(*ih) > data_size) {
            pr_debug("IPv4 overrun detected\n");
            goto buffer_overrun_check_failed;
        }

        if (ih->protocol == IPPROTO_IGMP)
            return false;
#if 0
        if (ih->protocol == IPPROTO_ICMP)
            return false;
#endif
        pr_debug("Filtered IPv4 protocol '%02hhx'\n", ih->protocol);
    }
    else
        pr_debug("Filtered ethernet protocol '%04hx'\n", proto);

buffer_overrun_check_failed:
    return true;
}

static netdev_tx_t
xmit_on_interface(struct sk_buff * skb, struct net_device * net_dev)
{
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_tx_dev * tx_dev = priv->dev->tx_dev;
    int err;

    netdev_dbg(net_dev, "Xmit\n");

    if (priv->dev->is_igmp_only && is_filtered(skb->data, skb->len)) {
        kfree_skb(skb);
        netdev_dbg(net_dev, "Discarded filtered tx frame\n");
        ++ priv->stats.tx_dropped;
        return NETDEV_TX_OK;
    }

    err = tx_dev->send(tx_dev, priv->id, skb->data, skb->len);
    if (err == -EAGAIN) {
        BUG_ON(priv->pending_tx_skb);

        netif_stop_queue(priv_to_net_dev(priv));

        priv->pending_tx_skb = skb;
        priv->tx_timeout_count = 0;

        mod_timer(&priv->tx_timer, jiffies + HFP_NET_TX_TIMEOUT);
    }
    else if (err < 0){
        netdev_dbg(net_dev, "Failed to send skb\n");
        ++ priv->stats.tx_errors;
        kfree_skb(skb);
    }
    else {
        priv->stats.tx_bytes += skb->len;
        ++ priv->stats.tx_packets;
        consume_skb(skb);
    }

    return NETDEV_TX_OK;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 6, 0)
#    define USE_5_6_TIMEOUT
#elif defined(RHEL_RELEASE_CODE)
#    if RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(8, 3)
#        define USE_5_6_TIMEOUT
#    endif
#endif

static void
#ifdef USE_5_6_TIMEOUT
on_interface_timeout(struct net_device * net_dev, unsigned int txqueue)
#else
on_interface_timeout(struct net_device * net_dev)
#endif
{
    netdev_dbg(net_dev, "Timeout\n");
}

static void
on_rx_packet(const uint8_t * data, uint32_t data_size, void * opaque)
{
    struct enyx_hfp_net_dev_priv * priv = opaque;
    struct net_device * net_dev = priv->dev->net_devs[priv->id];
    struct sk_buff * skb;

    /* Ethernet frame should contain at least ETH_ZLEN bytes */
    if (data_size < ETH_ZLEN) {
        ++ priv->stats.rx_errors;
        netdev_dbg(net_dev, "Discarded runt frame\n");
        return;
    }

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 19, 0)
    skb = napi_alloc_skb(&priv->napi, data_size + NET_IP_ALIGN);
#else
    skb = netdev_alloc_skb(net_dev, data_size + NET_IP_ALIGN);
#endif

    if (! skb) {
        ++ priv->stats.rx_dropped;
        netdev_dbg(net_dev, "Failed to allocate rx skb\n");
        return;
    }

    /* Ethernet header size is 14 bytes, hence
       copy data starting at offset 2 to ensure
       IP headers will be aligned on 4 bytes */
    skb_reserve(skb, NET_IP_ALIGN);

    memcpy(skb_put(skb, data_size), data, data_size);
    skb->protocol = eth_type_trans(skb, net_dev);

    ++ priv->stats.rx_packets;
    priv->stats.rx_bytes += data_size;

    netif_receive_skb(skb);
}

static int
poll_for_rx_data(struct napi_struct * napi, int budget)
{
    struct enyx_hfp_net_dev_priv * priv = napi_to_net_dev_priv(napi);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;
    uint16_t channel_id = priv->id;
    struct net_device * net_dev = priv->dev->net_devs[channel_id];
    int work_done = 0;

    while (! rx_dev->poll_one(rx_dev, channel_id, on_rx_packet, priv)
           && ++work_done < budget)
        continue;

    rx_dev->update_rd_ptr(rx_dev, channel_id);

    /* If we consumed all available packets, stop
       napi polling and re-arm interrupts */
    if (work_done < budget) {
        napi_complete(napi);
        arm_rx_interrupt(priv);
    }

    netdev_dbg(net_dev, "Polling complete\n");

    return work_done;
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 11, 0)
#   define USE_4_11_STATS_API
#elif defined(RHEL_RELEASE_CODE)
#   if RHEL_RELEASE_CODE  >= RHEL_RELEASE_VERSION(7, 5)
#       define USE_4_11_STATS_API
#   endif
#endif

#ifdef USE_4_11_STATS_API
static void
#else
static struct rtnl_link_stats64 *
#endif
get_interface_stats(struct net_device * net_dev,
                    struct rtnl_link_stats64 * stats)
{
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev *rx = priv->dev->rx_dev;

    memset(stats, 0, sizeof(*stats));
    stats->tx_errors = priv->stats.tx_errors;
    stats->tx_packets = priv->stats.tx_packets;
    stats->tx_bytes = priv->stats.tx_bytes;

    stats->rx_errors = priv->stats.rx_errors;
    stats->rx_packets = priv->stats.rx_packets;
    stats->rx_bytes = priv->stats.rx_bytes;

    if (rx->get_fifo_errors) {
        stats->rx_fifo_errors = rx->get_fifo_errors(rx, priv->id);
    }

    if (rx->get_errors) {
        stats->rx_crc_errors = rx->get_errors(rx, priv->id);
        stats->rx_errors += stats->rx_crc_errors;
    }

#ifndef USE_4_11_STATS_API
    return stats;
#endif
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 10, 0)
#   define USE_4_10_CHANGE_MTU
#endif

#ifndef USE_4_10_CHANGE_MTU
static int
change_mtu(struct net_device * net_dev, int new_mtu)
{
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);

    if (new_mtu < ETH_ZLEN || new_mtu > priv->max_mtu)
        return -EINVAL;

    net_dev->mtu = new_mtu;

    return 0;
}
#endif

static int
set_mac(struct net_device *netdev, void *p)
{
    struct enyx_hfp_net_dev_priv *priv = netdev_priv(netdev);
    struct enyx_hfp_net_dev *dev = priv->dev;
    struct sockaddr *addr = p;

    if (!is_valid_ether_addr(addr->sa_data))
        return -EADDRNOTAVAIL;

    memcpy(netdev->dev_addr, addr->sa_data, netdev->addr_len);

    if (dev->enable_filtering && (priv->whitelist_entry_count != 0))
        set_whitelist_entry(dev, priv->id, 0,
                            addr->sa_data, netdev->addr_len);

    return 0;
}

static int
get_whitelist_offset(struct enyx_hfp_net_dev *dev, uint16_t channel_id,
                     uint16_t * offset)
{
    enyx_hfp_write16(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, channel_id);
    if (enyx_hfp_wait_until_val_is(dev->parent,
                                   HFP_NET_DEVICE_INFO_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select interface page\n");
        goto interface_page_select_failed;
    }

    *offset = enyx_hfp_read16(dev->parent,
                              HFP_NET_DEVICE_WHITELIST_PAGE_OFFSET);

    return 0;

interface_page_select_failed:
    return -1;
}

static void
set_rx_mode(struct net_device *netdev)
{
    struct enyx_hfp_net_dev_priv *priv = netdev_priv(netdev);
    struct enyx_hfp_net_dev *dev = priv->dev;
    struct netdev_hw_addr *ha;
    size_t entries_count = 1;

    if (!dev->enable_filtering || (priv->whitelist_entry_count == 0))
        // Filtering feature is not supported or not enabled
        return;

    // Both UNI & MULTI are promiscuous
    if (netdev->flags & IFF_PROMISC) {
        set_ucast_promisc(dev, priv->id, true);
        set_mcast_promisc(dev, priv->id, true);
        goto disable_remaining_entries;
    }

    // Unicast
    if (entries_count + netdev_uc_count(netdev) <= priv->whitelist_entry_count) {
        set_ucast_promisc(dev, priv->id, false);
        netdev_for_each_uc_addr(ha, netdev)
            set_whitelist_entry(dev, priv->id, entries_count++,
                                ha->addr, ETH_ALEN);
    }
    else {
        netdev_dbg(netdev, "whitelist to small, fallback to unicast promiscuous");
        set_ucast_promisc(dev, priv->id, true);
    }

    // Multicast
    if (netdev->flags & IFF_ALLMULTI) {
        set_mcast_promisc(dev, priv->id, true);
        goto disable_remaining_entries;
    }

    if (entries_count + netdev_mc_count(netdev) > priv->whitelist_entry_count) {
        netdev_dbg(netdev, "whitelist to small, fallback to multicast promiscuous");
        set_mcast_promisc(dev, priv->id, true);
        goto disable_remaining_entries;
    }

    set_mcast_promisc(dev, priv->id, false);
    netdev_for_each_mc_addr(ha, netdev)
        set_whitelist_entry(dev, priv->id, entries_count++,
                            ha->addr, ETH_ALEN);

disable_remaining_entries:
    for (; entries_count != priv->whitelist_entry_count; ++entries_count)
        disable_whitelist_entry(dev, priv->id, entries_count);
}

#if defined(RHEL_RELEASE_CODE)
#   if RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(7, 5)
#       define USE_RH_7_5_CHANGE_MTU
#   endif
#endif

static const struct net_device_ops enyx_hfp_net_dev_ops = {
    .ndo_open = open_interface,
    .ndo_stop = close_interface,
    .ndo_start_xmit = xmit_on_interface,
    .ndo_tx_timeout = on_interface_timeout,
    .ndo_get_stats64 = get_interface_stats,
    .ndo_set_rx_mode = set_rx_mode,
    .ndo_set_mac_address = set_mac,
#ifndef USE_4_10_CHANGE_MTU
#   ifdef USE_RH_7_5_CHANGE_MTU
    .ndo_change_mtu_rh74 = change_mtu,
#   else
    .ndo_change_mtu = change_mtu,
#   endif
#endif
};

static ssize_t
port_usage_show(struct device * device,
           struct device_attribute * attr,
           char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    char usage[IFNAMSIZ];

    memset(usage, 0, sizeof(usage));
    if (priv->dev->get_usage)
        priv->dev->get_usage(priv->dev, priv->id, usage, sizeof(usage));

    return scnprintf(buf, PAGE_SIZE, "%s\n", usage);
}

static struct device_attribute enyx_hfp_net_dev_attr_port_usage
        = __ATTR_RO(port_usage);

static ssize_t
port_id_show(struct device * device,
             struct device_attribute * attr,
             char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);

    return scnprintf(buf, PAGE_SIZE, "%hu\n", priv->port_id);
}

static struct device_attribute enyx_hfp_net_dev_attr_port_id
        = __ATTR_RO(port_id);

static ssize_t
backpressure_count_show(struct device * device,
                        struct device_attribute * attr,
                        char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;

    if (rx_dev->get_backpressure_count)
        return scnprintf(buf, PAGE_SIZE, "%zu\n",
                         rx_dev->get_backpressure_count(rx_dev,
                                                        priv->id));
    return scnprintf(buf, PAGE_SIZE, "\n");
}

static struct device_attribute enyx_hfp_net_dev_attr_backpressure_count
        = __ATTR_RO(backpressure_count);

static ssize_t
packet_count_show(struct device * device,
                  struct device_attribute * attr,
                  char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;

    if (rx_dev->get_packet_count)
        return scnprintf(buf, PAGE_SIZE, "%zu\n",
                         rx_dev->get_packet_count(rx_dev,
                                                  priv->id));
    return scnprintf(buf, PAGE_SIZE, "\n");
}

static struct device_attribute enyx_hfp_net_dev_attr_packet_count
        = __ATTR_RO(packet_count);

static ssize_t
errors_show(struct device * device,
            struct device_attribute * attr,
            char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);

    return scnprintf(buf, PAGE_SIZE, "%zu\n",
            rx_netdev_get_errors(priv));
}

static struct device_attribute enyx_hfp_net_dev_attr_errors
        = __ATTR_RO(errors);

static ssize_t
fifo_errors_show(struct device * device,
                 struct device_attribute * attr,
                 char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;

    if (rx_dev->get_fifo_errors)
        return scnprintf(buf, PAGE_SIZE, "%zu\n",
                rx_dev->get_fifo_errors(rx_dev,
                    priv->id));
    return scnprintf(buf, PAGE_SIZE, "\n");
}

static struct device_attribute enyx_hfp_net_dev_attr_fifo_errors
        = __ATTR_RO(fifo_errors);

static ssize_t
truncated_count_show(struct device * device,
                 struct device_attribute * attr,
                 char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;

    if (rx_dev->get_truncated_count)
        return scnprintf(buf, PAGE_SIZE, "%zu\n",
                rx_dev->get_truncated_count(rx_dev,
                    priv->id));
    return scnprintf(buf, PAGE_SIZE, "\n");
}

static struct device_attribute enyx_hfp_net_dev_attr_truncated_count
        = __ATTR_RO(truncated_count);

static ssize_t
crc_errors_show(struct device * device,
                struct device_attribute * attr,
                char * buf)
{
    struct net_device * net_dev = to_net_dev(device);
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
    struct enyx_hfp_rx_dev * rx_dev = priv->dev->rx_dev;

    if (rx_dev->get_errors)
        return scnprintf(buf, PAGE_SIZE, "%zu\n",
                rx_dev->get_errors(rx_dev,
                    priv->id));
    return scnprintf(buf, PAGE_SIZE, "\n");
}

static struct device_attribute enyx_hfp_net_dev_attr_crc_errors
        = __ATTR_RO(crc_errors);

static struct attribute * enyx_hfp_net_dev_attrs[] = {
    &enyx_hfp_net_dev_attr_port_usage.attr,
    &enyx_hfp_net_dev_attr_port_id.attr,
    &enyx_hfp_net_dev_attr_backpressure_count.attr,
    &enyx_hfp_net_dev_attr_packet_count.attr,
    &enyx_hfp_net_dev_attr_errors.attr,
    &enyx_hfp_net_dev_attr_fifo_errors.attr,
    &enyx_hfp_net_dev_attr_truncated_count.attr,
    &enyx_hfp_net_dev_attr_crc_errors.attr,
    NULL,
};

static const struct attribute_group enyx_hfp_net_dev_group = {
    .attrs = enyx_hfp_net_dev_attrs,
};

static uint64_t
get_mac_addr(struct enyx_hfp_net_dev * dev, uint16_t i)
{
    uint64_t result = 0;

    if (down_interruptible(&dev->parent->lock))
        goto interface_page_select_lock_failed;

    enyx_hfp_write32(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, i);
    if (enyx_hfp_wait_until_val_is(dev->parent,
                              HFP_NET_DEVICE_INFO_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select interface page\n");
        goto interface_page_select_failed;
    }

    result = enyx_hfp_read64(dev->parent, HFP_NET_DEVICE_INFO_MAC_ADDRESS);

interface_page_select_failed:
    up(&dev->parent->lock);
interface_page_select_lock_failed:
    return result;
}

/* ENYX_HFP watchdog */

static bool update_carrier(struct enyx_hfp_net_dev_priv * priv)
{
    struct enyx_hfp_net_dev * dev = priv->dev;
    struct net_device * net_dev = dev->net_devs[priv->id];
    bool carrier = false;
    if (!dev->get_carrier)
        return true;

    carrier = dev->get_carrier(dev, priv->id);
    if (carrier != netif_carrier_ok(net_dev)) {
        if (carrier)
            netif_carrier_on(net_dev);
        else
            netif_carrier_off(net_dev);
    }
    return carrier;
}

static void enyx_hfp_watchdog(struct work_struct *work)
{
    struct enyx_hfp_net_dev_priv * priv = container_of(
            work,
            struct enyx_hfp_net_dev_priv,
            watchdog_task.work
    );
    update_carrier(priv);
    schedule_delayed_work(&priv->watchdog_task, 1 * HZ);
}

/* ENYX_HFP ethtool */

enum { NETDEV_STATS, ENYX_HFP_RX_STATS, ENYX_HFP_FILTER_STATS };

struct enyx_hfp_stats {
    char stat_string[ETH_GSTRING_LEN];
    int type;
    size_t (*get)(struct enyx_hfp_net_dev_priv *);
};

#define ENYX_HFP_RX_ATTR(str) { \
    .stat_string = "rx_"#str, \
    .type = ENYX_HFP_RX_STATS, \
    .get = rx_get_##str }

#define ENYX_HFP_RX_NETDEV_ATTR(str) { \
    .stat_string = "rx_"#str, \
    .type = NETDEV_STATS, \
    .get = rx_netdev_get_##str }

#define ENYX_HFP_RX_FILTER_ATTR(str) { \
    .stat_string = "rx_filter_"#str, \
    .type = ENYX_HFP_FILTER_STATS, \
    .get = rx_filter_get_##str }

static const struct enyx_hfp_stats enyx_hfp_gstrings_stats[] = {
    ENYX_HFP_RX_ATTR(packet_count),
    ENYX_HFP_RX_ATTR(backpressure_count),
    ENYX_HFP_RX_NETDEV_ATTR(errors),
    ENYX_HFP_RX_ATTR(fifo_errors),
    ENYX_HFP_RX_ATTR(truncated_count),
    ENYX_HFP_RX_ATTR(crc_errors),
    ENYX_HFP_RX_FILTER_ATTR(input_count),
    ENYX_HFP_RX_FILTER_ATTR(drop_count),
    ENYX_HFP_RX_FILTER_ATTR(unicast_promisc_en),
    ENYX_HFP_RX_FILTER_ATTR(multicast_promisc_en),
    ENYX_HFP_RX_FILTER_ATTR(whitelist_capacity),
    ENYX_HFP_RX_FILTER_ATTR(whitelist_usage),
};

#define ENYX_HFP_STATS_LEN ARRAY_SIZE(enyx_hfp_gstrings_stats)

static void enyx_hfp_get_ethtool_stats(struct net_device *netdev,
                    struct ethtool_stats __always_unused *stats,
                    u64 *data) {
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(netdev);

    int i;
    for (i = 0; i < ENYX_HFP_STATS_LEN; i++) {
        switch (enyx_hfp_gstrings_stats[i].type) {
        case NETDEV_STATS:
        case ENYX_HFP_RX_STATS:
        case ENYX_HFP_FILTER_STATS:
            data[i] = enyx_hfp_gstrings_stats[i].get(priv);
            break;
        default:
            data[i] = 0;
            continue;
        }
    }
}

static int
enyx_hfp_get_sset_count(struct net_device __always_unused *netdev,
                        int sset)
{
    switch (sset) {
    case ETH_SS_STATS:
        return ENYX_HFP_STATS_LEN;
    default:
        return -EOPNOTSUPP;
    }
}

static void
enyx_hfp_get_strings(struct net_device __always_unused *netdev,
                     u32 stringset, u8 *data)
{
    u8 *p = data;
    int i;

    switch (stringset) {
    case ETH_SS_STATS:
        for (i = 0; i < ENYX_HFP_STATS_LEN; ++i) {
            memcpy(p, enyx_hfp_gstrings_stats[i].stat_string,
                   ETH_GSTRING_LEN);
            p += ETH_GSTRING_LEN;
        }
        break;
    }
}

static void
enyx_hfp_get_drvinfo(struct net_device *netdev,
                     struct ethtool_drvinfo *drvinfo)
{
    struct enyx_hfp_net_dev_priv *priv = netdev_priv(netdev);
    struct enyx_hfp_net_dev *dev = priv->dev;
    strlcpy(drvinfo->driver, THIS_MODULE->name,
            sizeof(drvinfo->driver));
    strlcpy(drvinfo->version, ENYX_HFP_MODULES_VERSION,
            sizeof(drvinfo->version));
    snprintf(drvinfo->bus_info, sizeof(drvinfo->bus_info), "%d",
            dev->parent->bus->index);
}

static uint32_t
enyx_hfp_get_link(struct net_device *netdev)
{
    struct enyx_hfp_net_dev_priv * priv = netdev_priv(netdev);
    return update_carrier(priv);
}

static const struct ethtool_ops enyx_hfp_ethtool_ops = {
    .get_drvinfo = enyx_hfp_get_drvinfo,
    .get_link = enyx_hfp_get_link,
    .get_strings = enyx_hfp_get_strings,
    .get_ethtool_stats = enyx_hfp_get_ethtool_stats,
    .get_sset_count = enyx_hfp_get_sset_count,
};

/* net_dev init */

#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 1, 0)
#    define USE_WEIGHTLESS_NAPI_ADD
#elif defined(RHEL_RELEASE_CODE)
#    if (RHEL_MAJOR == 8 && RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(8, 8))
#        define USE_WEIGHTLESS_NAPI_ADD
#    elif RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(9, 2)
#        define USE_WEIGHTLESS_NAPI_ADD
#    endif
#endif

static int
init_net_dev(struct enyx_hfp_net_dev * dev, uint16_t i)
{
    struct net_device * net_dev;
    struct enyx_hfp_net_dev_priv * priv;
    uint64_t mac_addr;

    net_dev = dev->net_devs[i] = alloc_etherdev(sizeof(*priv));
    if (! net_dev) {
        dev_err(&dev->parent->device, "Can't allocate ether dev\n");
        goto alloc_etherdev_failed;
    }

    SET_NETDEV_DEV(net_dev, &dev->parent->device);

    mac_addr = get_mac_addr(dev, i);
    net_dev->dev_addr[0] = mac_addr >> 40;
    net_dev->dev_addr[1] = mac_addr >> 32;
    net_dev->dev_addr[2] = mac_addr >> 24;
    net_dev->dev_addr[3] = mac_addr >> 16;
    net_dev->dev_addr[4] = mac_addr >> 8;
    net_dev->dev_addr[5] = mac_addr;

    if (! is_valid_ether_addr(net_dev->dev_addr)) {
        dev_err(&dev->parent->device,
                "Interface %d MAC %pM is invalid\n",
                i, net_dev->dev_addr);
        goto mac_check_failed;
    }
    dev_dbg(&dev->parent->device, "Interface %d MAC is %pM\n",
            i, net_dev->dev_addr);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 15, 0)
    net_dev->dev_port = i;
#endif
    net_dev->netdev_ops = &enyx_hfp_net_dev_ops;

    /* TODO: Get max_mtu per channel at this point instead of global value */
    priv = netdev_priv(net_dev);
    priv->max_mtu = min(dev->rx_dev->get_mtu(dev->rx_dev, i),
                       dev->tx_dev->get_mtu(dev->tx_dev, i));
    dev_dbg(&dev->parent->device, "Detected max MTU for device %d is %u\n", i,
            priv->max_mtu);
#ifdef USE_4_10_CHANGE_MTU
    net_dev->min_mtu = 64;
    net_dev->max_mtu = priv->max_mtu;
#endif
    net_dev->watchdog_timeo = 5 * HZ;
    strlcpy(net_dev->name, "eth%d", sizeof(net_dev->name));
    net_dev->sysfs_groups[0] = &enyx_hfp_net_dev_group;

    priv->id = i;
    priv->port_id = mac_addr - get_mac_addr(dev, 0);
    priv->dev = dev;
    memset(&priv->stats, 0, sizeof(priv->stats));
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 14, 0)
    timer_setup(&priv->tx_timer, on_tx_buffer_full, 0);
#else
    setup_timer(&priv->tx_timer, on_tx_buffer_full,
                (unsigned long)priv);
#endif
    priv->pending_tx_skb = NULL;
    priv->is_open = false;
#ifdef USE_WEIGHTLESS_NAPI_ADD
    netif_napi_add(net_dev, &priv->napi,
                   poll_for_rx_data);
#else
    netif_napi_add(net_dev, &priv->napi,
                   poll_for_rx_data, NAPI_POLL_WEIGHT);
#endif

    if (register_netdev(net_dev) < 0) {
        dev_err(&dev->parent->device, "Failed to register interface %d\n", i);
        goto register_netdev_failed;
    }

    net_dev->ethtool_ops = &enyx_hfp_ethtool_ops;
    INIT_DELAYED_WORK(&priv->watchdog_task, enyx_hfp_watchdog);
    schedule_delayed_work(&priv->watchdog_task, 1);

    netdev_info(net_dev, "Created\n");

    return 0;

register_netdev_failed:
mac_check_failed:
    free_netdev(net_dev);
alloc_etherdev_failed:
    return -EIO;
}

static int
init_net_devs(struct enyx_hfp_net_dev * dev)
{
    uint16_t i = 0, e = dev->channel_count;

    for (; i != e; ++i)
        if (init_net_dev(dev, i) < 0)
            goto init_net_dev_failed;

    return 0;

init_net_dev_failed:
    while (i) {
        struct net_device * net_dev = dev->net_devs[-- i];
        unregister_netdev(net_dev);
        free_netdev(net_dev);
    }
    return -1;
}

static const struct enyx_hfp_device_id enyx_hfp_rx_id = {
    HFP_DEVICE_ID(262)
};

static int
init_rx_dev(struct enyx_hfp_net_dev * dev)
{
    struct enyx_hfp_device * rx;
    uint16_t i;

    rx = enyx_hfp_find_depth_first(dev->parent, &enyx_hfp_rx_id, 0);
    if (! rx) {
        dev_err(&dev->parent->device, "Failed to find rx\n");
        goto enyx_hfp_find_rx_failed;
    }

    dev->rx_dev = enyx_hfp_rx_dev_create(rx,
                                    on_rx_interrupt, dev,
                                    HFP_NET_RX_BUFFER_SIZE_SHIFT,
                                    dev->coherency_line_size);
    if (! dev->rx_dev)
        goto enyx_hfp_rx_dev_create;

    if (dev->rx_dev->get_channel_count(dev->rx_dev) != dev->channel_count) {
        dev_err(&rx->device, "Unexpected rx channel count\n");
        goto rx_channel_count_check_failed;
    }

    for (i = 0; i != dev->channel_count; ++i)
        if (dev->rx_dev->alloc_dma_data_buffer(dev->rx_dev, i) != 0)
            goto rx_alloc_dma_data_buffer_failed;

    return 0;

rx_alloc_dma_data_buffer_failed:
    while (i)
        dev->rx_dev->free_dma_data_buffer(dev->rx_dev, i);
rx_channel_count_check_failed:
    enyx_hfp_rx_dev_destroy(dev->rx_dev);
enyx_hfp_rx_dev_create:
enyx_hfp_find_rx_failed:
    return -EIO;
}

static const struct enyx_hfp_device_id enyx_hfp_tx_id = {
    HFP_DEVICE_ID(263)
};

static int
init_tx_dev(struct enyx_hfp_net_dev * dev)
{
    struct enyx_hfp_device * tx;

    tx = enyx_hfp_find_depth_first(dev->parent, &enyx_hfp_tx_id, 0);
    if (! tx) {
        dev_err(&dev->parent->device, "Failed to find tx\n");
        goto enyx_hfp_find_tx_failed;
    }

    dev->tx_dev = enyx_hfp_tx_dev_create(tx, dev->coherency_line_size);
    if (! dev->tx_dev)
        goto enyx_hfp_tx_dev_create;

    if (dev->tx_dev->get_channel_count(dev->tx_dev) != dev->channel_count) {
        dev_err(&tx->device, "Unexpected tx channel count\n");
        goto tx_channel_count_check_failed;
    }

    return 0;

tx_channel_count_check_failed:
    enyx_hfp_tx_dev_destroy(dev->tx_dev);
enyx_hfp_tx_dev_create:
enyx_hfp_find_tx_failed:
    return -EIO;
}

static void
init_filtering(struct enyx_hfp_net_dev *dev)
{
    uint16_t dev_id;
    struct net_device *netdev;
    struct enyx_hfp_net_dev_priv *priv;

    if (!dev->enable_filtering)
        return;

    for (dev_id = 0; dev_id != dev->channel_count; ++dev_id) {
        netdev = dev->net_devs[dev_id];
        priv = netdev_priv(netdev);

        dev->enable_filtering(dev, dev_id);

        priv->whitelist_page_offset = 0;
        priv->whitelist_entry_count = 0;

        if (get_whitelist_offset(dev, dev_id,
                                 &priv->whitelist_page_offset) < 0) {
            dev_err(&dev->parent->device,
                    "cannot get whitelist offset for channel %d",
                    dev_id);
            continue;
        }

        priv->whitelist_entry_count = get_whitelist_count(dev, dev_id);

        if (priv->whitelist_entry_count != 0) {
            set_whitelist_entry(dev, dev_id, 0,
                                netdev->dev_addr, netdev->addr_len);
            set_rx_mode(netdev);
            netdev_dbg(netdev, "device filtering enabled");
        } else {
            /*
             * Ensure ucast/mcast promisc status in the priv struct is set to
             * true to have the right values in ethtool stats.
             */
            set_ucast_promisc(dev, dev_id, true);
            set_mcast_promisc(dev, dev_id, true);
            netdev_dbg(netdev,
                       "device has no whitelist, filter promiscuous mode enabled");
        }
    }
}

static size_t
get_usage_v0_2(struct enyx_hfp_net_dev * dev,
               uint16_t channel_id,
               char * buffer, size_t buffer_size)
{
    size_t count = 0;

    if (down_interruptible(&dev->parent->lock))
        return 0ULL;

    enyx_hfp_write8(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, channel_id);

    count = enyx_hfp_read_str_unlocked(dev->parent,
                                  HFP_NET_DEVICE_INFO_USAGE_STR,
                                  HFP_NET_DEVICE_INFO_USAGE_STR_INDEX,
                                  HFP_NET_DEVICE_INFO_PAGE_READY,
                                  buffer, buffer_size);

    up(&dev->parent->lock);
    return count;
}

static bool
get_carrier_v0_4(struct enyx_hfp_net_dev * dev, uint16_t i)
{
    uint8_t result = 0;

    if (down_interruptible(&dev->parent->lock))
        goto interface_page_select_lock_failed;

    enyx_hfp_write32(dev->parent, HFP_NET_DEVICE_INFO_PAGE_ID, i);
    if (enyx_hfp_wait_until_val_is(dev->parent,
                              HFP_NET_DEVICE_INFO_PAGE_READY, 1) < 0) {
        dev_err(&dev->parent->device, "Failed to select interface page\n");
        goto interface_page_select_failed;
    }

    result = enyx_hfp_read8(dev->parent, HFP_NET_DEVICE_INFO_STATUS);

interface_page_select_failed:
    up(&dev->parent->lock);
interface_page_select_lock_failed:
    return result & 1;
}

static void
bind_methods(struct enyx_hfp_net_dev * dev)
{
    if (enyx_hfp_get_device_version(dev->parent) >= HFP_COMPUTE_VERSION(0, 2))
        dev->get_usage = get_usage_v0_2;
    if (enyx_hfp_get_device_version(dev->parent) >= HFP_COMPUTE_VERSION(0, 4))
        dev->get_carrier = get_carrier_v0_4;

    if (enyx_hfp_get_device_version(dev->parent) >= HFP_COMPUTE_VERSION(0, 5)) {
        dev->enable_filtering = enable_filtering;
    }
}

struct enyx_hfp_net_dev *
enyx_hfp_net_dev_create(struct enyx_hfp_device * parent, bool is_igmp_only)
{
    struct enyx_hfp_net_dev * dev;
    uint16_t channel_count;

    if (! get_device(&parent->device))
        goto get_device_failed;

    if (parent->size < 8 * 4) {
        dev_err(&parent->device,
                "Reported device memory range smaller than expected\n");
        goto size_check_failed;
    }

    channel_count = enyx_hfp_read32(parent, HFP_NET_DEVICE_COUNT);
    dev_dbg(&parent->device, "Found %u channels\n", channel_count);

    if (! channel_count)
        goto channel_count_check_failed;

    dev = kzalloc(sizeof(*dev) + channel_count * sizeof(*dev->net_devs),
                  GFP_KERNEL);
    if (! dev) {
        dev_err(&parent->device, "Can't allocate net\n");
        goto kzalloc_failed;
    }

    dev->parent = parent;
    dev->coherency_line_size = cache_line_size();
    dev->channel_count = channel_count;

    dev->is_igmp_only = is_igmp_only;

    bind_methods(dev);

    if (init_rx_dev(dev) < 0)
        goto init_rx_dev_failed;

    if (init_tx_dev(dev) < 0)
        goto init_tx_dev_failed;

    if (init_net_devs(dev) < 0)
        goto init_net_devs_failed;

    init_filtering(dev);

    return dev;

init_net_devs_failed:
    enyx_hfp_tx_dev_destroy(dev->tx_dev);
init_tx_dev_failed:
    enyx_hfp_rx_dev_destroy(dev->rx_dev);
init_rx_dev_failed:
    kfree(dev);
kzalloc_failed:
channel_count_check_failed:
size_check_failed:
    put_device(&parent->device);
get_device_failed:
    return NULL;
}
EXPORT_SYMBOL(enyx_hfp_net_dev_create);

static void
destroy_net_devs(struct enyx_hfp_net_dev * dev)
{
    while (dev->channel_count) {
        struct net_device * net_dev = dev->net_devs[-- dev->channel_count];
        struct enyx_hfp_net_dev_priv * priv = netdev_priv(net_dev);
#if 0
        close_interface(net_dev);
#endif
        cancel_delayed_work_sync(&priv->watchdog_task);
        unregister_netdev(net_dev);
        free_netdev(net_dev);
    }
}

void
enyx_hfp_net_dev_destroy(struct enyx_hfp_net_dev * dev)
{
    destroy_net_devs(dev);

    enyx_hfp_rx_dev_destroy(dev->rx_dev);
    enyx_hfp_tx_dev_destroy(dev->tx_dev);

    put_device(&dev->parent->device);

    kfree(dev);
}
EXPORT_SYMBOL(enyx_hfp_net_dev_destroy);
