/*
 * HFP mm module.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/module.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_char_device.h"
#include "enyx_hfp_bus.h"
#include "enyx_hfp_mm_usr.h"

static struct enyx_hfp_char_device_region * enyx_hfp_mm_char_device_region;

#define HFP_MM_ID 17

static DEFINE_HFP_DEVICE_TABLE(enyx_hfp_mm_device_table) = {
    {HFP_DEVICE_ID(HFP_MM_ID)},
    {0},
};
MODULE_ALIAS_HFP_ID(HFP_MM_ID);

struct enyx_hfp_drvdata
{
    struct enyx_hfp_mm_usr * mm;
};

#define HFP_MM_ALIGNMENT_REG_INDEX (8)

static bool
is_mappable(const struct enyx_hfp_device * dev)
{
    uint64_t offset = dev->bus_offset;
    offset += HFP_MM_ALIGNMENT_REG_INDEX << HFP_BUS_ADDR_SHIFT;
    return readl(dev->bus->virt_addr + offset) >= PAGE_SHIFT;
}

static bool
is_parent_mappable(struct enyx_hfp_device * dev)
{
    while ((dev = enyx_hfp_get_device_parent(dev)))
        if (dev->id.hardware_id == HFP_MM_ID && is_mappable(dev))
            return true;

    return false;
}

static int
enyx_hfp_device_probe(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_drvdata * drvdata;
    int err;

    /* Skip non aligned mm devices */
    if (! is_mappable(dev)) {
        err = -ENODEV;
        goto err_is_mappable_failed;
    }

    if (is_parent_mappable(dev)) {
        err = -ENODEV;
        dev_err(&dev->device,
                "Can't bind to a child of an already bound mm\n");
        goto err_is_parent_mappable_failed;
    }

    drvdata = kzalloc(sizeof(*drvdata), GFP_KERNEL);
    if (! drvdata) {
        err = -ENOMEM;
        dev_err(&dev->device, "Can't allocate enyx_hfp mm struct");
        goto err_enyx_hfp_mm_drvdata_alloc;
    }
    enyx_hfp_set_drvdata(dev, drvdata);

    drvdata->mm = create_enyx_hfp_mm_usr(dev, enyx_hfp_mm_char_device_region);
    if (! drvdata->mm) {
        err = -ENODEV;
        goto err_create_enyx_hfp_mm_usr;
    }

    return 0;

err_create_enyx_hfp_mm_usr:
    kfree(drvdata);
err_enyx_hfp_mm_drvdata_alloc:
err_is_parent_mappable_failed:
err_is_mappable_failed:
    return err;
}

static void
enyx_hfp_device_remove(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_drvdata * drvdata = enyx_hfp_get_drvdata(dev);

    destroy_enyx_hfp_mm_usr(drvdata->mm);

    kfree(drvdata);
}

static struct enyx_hfp_driver enyx_hfp_mm_driver = {
    .name = THIS_MODULE->name,
    .id_table = enyx_hfp_mm_device_table,
    .probe = enyx_hfp_device_probe,
    .remove = enyx_hfp_device_remove,
};

static int __init
init_this_module(void)
{
    int err;

    pr_info("%s %s <support@enyx.com>\n",
            MODULE_NAME, ENYX_HFP_MODULES_VERSION);

    enyx_hfp_mm_char_device_region = enyx_hfp_char_device_region_create(MODULE_NAME);
    if (! enyx_hfp_mm_char_device_region) {
        err = -EINVAL;
        goto err_alloc_chrdev_region;
    }

    err = enyx_hfp_mm_class_register();
    if (err < 0)
        goto err_enyx_hfp_mm_class_register;

    err = enyx_hfp_register_driver(&enyx_hfp_mm_driver);
    if (err < 0)
        goto err_enyx_hfp_register_driver;

    return 0;

err_enyx_hfp_register_driver:
    enyx_hfp_mm_class_unregister();
err_enyx_hfp_mm_class_register:
    enyx_hfp_char_device_region_destroy(enyx_hfp_mm_char_device_region);
err_alloc_chrdev_region:
    return err;
}

module_init(init_this_module);

static void __exit
exit_this_module(void)
{
    enyx_hfp_unregister_driver(&enyx_hfp_mm_driver);

    enyx_hfp_mm_class_unregister();

    enyx_hfp_char_device_region_destroy(enyx_hfp_mm_char_device_region);

    pr_debug("%s Exited\n", MODULE_NAME);
}

module_exit(exit_this_module);

MODULE_AUTHOR("David Keller <david.keller@enyx.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Hfp mm module");
MODULE_VERSION(ENYX_HFP_MODULES_VERSION);

