/*
 * HFP char device region implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_irq_list.h"

#include <linux/printk.h>
#include <linux/slab.h>

#include "enyx_hfp_common.h"

struct enyx_hfp_irq_list_entry
{
    int irq;
    int index;
};

struct enyx_hfp_irq_list
{
    size_t capacity;
    size_t size;
    struct enyx_hfp_irq_list_entry irqs[];
};

struct enyx_hfp_irq_list *
enyx_hfp_irq_list_create(size_t capacity)
{
    struct enyx_hfp_irq_list * l;

    l = kzalloc(sizeof(*l) + capacity * sizeof(*l->irqs), GFP_KERNEL);
    if (! l) {
        pr_err("%s: Can't allocate irq_list\n", MODULE_NAME);
        goto err_kmalloc;
    }

    l->capacity = capacity;
    l->size = 0ULL;

err_kmalloc:
    return l;
}
EXPORT_SYMBOL(enyx_hfp_irq_list_create);

void
enyx_hfp_irq_list_destroy(struct enyx_hfp_irq_list * irq_list)
{
    kfree(irq_list);
}
EXPORT_SYMBOL(enyx_hfp_irq_list_destroy);

int
enyx_hfp_irq_list_get(struct enyx_hfp_irq_list * irq_list,
                 int * irq, int * index)
{
    if (! irq_list)
        goto irq_list_ptr_test_failed;

    if (irq_list->size == 0) {
        pr_err("%s: Can't get irq from empty irq_list\n", MODULE_NAME);
        goto irq_list_empty_test_failed;
    }

    -- irq_list->size;
    *irq = irq_list->irqs[irq_list->size].irq;
    *index = irq_list->irqs[irq_list->size].index;

    return 0;

irq_list_empty_test_failed:
irq_list_ptr_test_failed:
    return -EINVAL;
}
EXPORT_SYMBOL(enyx_hfp_irq_list_get);

int
enyx_hfp_irq_list_put(struct enyx_hfp_irq_list * irq_list,
                 int irq, int index)
{
    if (! irq_list)
        goto irq_list_ptr_test_failed;

    if (irq_list->size == irq_list->capacity) {
        pr_err("%s: Can't put irq to full irq_list\n", MODULE_NAME);
        goto irq_list_full_test_failed;
    }

    irq_list->irqs[irq_list->size].irq = irq;
    irq_list->irqs[irq_list->size].index = index;

    ++ irq_list->size;

    return 0;

irq_list_full_test_failed:
irq_list_ptr_test_failed:
    return -EINVAL;
}
EXPORT_SYMBOL(enyx_hfp_irq_list_put);

