/*
 * HFP char device region implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_char_device.h"

#include <linux/kernel.h>
#include <linux/printk.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/fs.h>

#include "enyx_hfp_common.h"

enum { FIRST_MINOR = 0 };
enum { MAX_CHAR_DEVICES = 128 };

struct enyx_hfp_char_device_region
{
    dev_t first_chrdev;
    DECLARE_BITMAP(minor_used, MAX_CHAR_DEVICES);
};

struct enyx_hfp_char_device_region *
enyx_hfp_char_device_region_create(const char * name)
{
    struct enyx_hfp_char_device_region * region;

    region = kzalloc(sizeof(*region), GFP_KERNEL);
    if (! region) {
        pr_err("%s: Can't allocate char device region\n", MODULE_NAME);
        goto err_kmalloc;
    }

    if (alloc_chrdev_region(&region->first_chrdev,
                            FIRST_MINOR,
                            MAX_CHAR_DEVICES,
                            name) < 0)
        goto err_alloc_chrdev_region;

    return region;

err_alloc_chrdev_region:
    kfree(region);
err_kmalloc:
    return NULL;
}
EXPORT_SYMBOL(enyx_hfp_char_device_region_create);

void
enyx_hfp_char_device_region_destroy(struct enyx_hfp_char_device_region * region)
{
    /* Ensure no char device dev number is still allocated */
    BUG_ON(find_first_bit(region->minor_used,
                          MAX_CHAR_DEVICES) != MAX_CHAR_DEVICES);

    unregister_chrdev_region(region->first_chrdev, MAX_CHAR_DEVICES);

    kfree(region);
}
EXPORT_SYMBOL(enyx_hfp_char_device_region_destroy);

dev_t
enyx_hfp_char_device_get(struct enyx_hfp_char_device_region * region)
{
    const int minor = find_first_zero_bit(region->minor_used,
                                          MAX_CHAR_DEVICES);
    BUG_ON(minor == MAX_CHAR_DEVICES);
    set_bit(minor, region->minor_used);

    return MKDEV(MAJOR(region->first_chrdev), minor);
}
EXPORT_SYMBOL(enyx_hfp_char_device_get);

void
enyx_hfp_char_device_put(struct enyx_hfp_char_device_region * region,
                    const dev_t dev)
{
    BUG_ON(MAJOR(dev) != MAJOR(region->first_chrdev));
    clear_bit(MINOR(dev), region->minor_used);
}
EXPORT_SYMBOL(enyx_hfp_char_device_put);

