#pragma once
#include <stdarg.h>
#include <enyx/log/symbol_visibility.h>

/**
 * Supported log levels.
 */
typedef enum {
    /** Used in case of critical errors, expect an abort right afterwards. */
    ENYX_LOG_ERROR,
    /** Used for non fatal errors */
    ENYX_LOG_WARNING,
    /** Useful non-error informations */
    ENYX_LOG_INFO, 
    /** Debug messages only */
    ENYX_LOG_DEBUG,
} enyx_log_level;

/**
 * Represent a log handler that is called on each log message.
 *
 * @since 4.0.0
 */
typedef void
(*enyx_log_handler)(enyx_log_level level,
                    char const * filename,
                    unsigned int line_number,
                    char const * function_name,
                    char const * format,
                    va_list args);

/**
 * Change the log handler.
 * 
 * Pass a NULL handler to reset the log handler to default.
 *
 * @param new_handler The new handler.
 * @return The old log handler.
 */
ENYX_LOG_C_SYMBOL_VISIBILITY enyx_log_handler
enyx_log_set_handler(enyx_log_handler new_handler);

/**
 * Change the current log level.
 *
 * @note The log level can also be set using the environment variable
 * "ENYX_LOG_LEVEL" with the following strings "ERROR", "WARNING", "INFO",
 * "DEBUG". If set, the environment variable has priority over programatically
 * set log level.
 *
 * @param new_level The new log level.
 * @return The old log level (environment variable being set or not has *no*
 *         impact on this value).
 */
ENYX_LOG_C_SYMBOL_VISIBILITY enyx_log_level
enyx_log_set_log_level(enyx_log_level new_level);

/**
 * Log a message
 *
 * @note Use level specific macros instead of this function.
 *
 * @param level The log level.
 * @param file The file that emits the message.
 * @param line line number that emits this message.
 * @param function function that emits this message.
 * @param format Message format in printf style (\n not needed).
 */
ENYX_LOG_C_SYMBOL_VISIBILITY void
enyx_log_log(enyx_log_level level,
             char const * file,
             unsigned int line,
             char const * function,
             char const * format, ...);

/**
 * @def enyx_log_error
 *
 * Log a fatal error message.
 */
#define enyx_log_error(...) enyx_log_log(ENYX_LOG_ERROR, \
                                         __FILE__, \
                                         __LINE__, \
                                         __func__, \
                                         __VA_ARGS__)

/**
 * @def enyx_log_warning.
 *
 * Log a non fatal error message.
 */
#define enyx_log_warning(...) enyx_log_log(ENYX_LOG_WARNING, \
                                           __FILE__, \
                                           __LINE__, \
                                           __func__, \
                                           __VA_ARGS__)

/**
 * @def enyx_log_info
 *
 * Log a non-error message.
 */
#define enyx_log_info(...) enyx_log_log(ENYX_LOG_INFO, \
                                        __FILE__, \
                                        __LINE__, \
                                        __func__, \
                                        __VA_ARGS__)

/**
 * @def enyx_log_debug
 *
 * Log a debug message.
 */
#ifndef NDEBUG
#define enyx_log_debug(...) enyx_log_log(ENYX_LOG_DEBUG, \
                                         __FILE__, \
                                         __LINE__, \
                                         __func__, \
                                         __VA_ARGS__)
#else
#define enyx_log_debug(...) {}
#endif
