#include <cstdlib>
#include <iostream>
#include <sstream>

#include <Argos/Argos.hpp>

#include <enyx/log/log.h>

#include <enyx/hw/accelerator_descriptor.hpp>

#include <enyx/fix/engine.hpp>
#include <enyx/fix/fix_model.hpp>

static auto
find_accelerator(std::string const& arg)
{
    enyx::hw::filter filter;
    try {
        std::size_t end;
        std::uint32_t acc_id = std::stol(arg, &end, 0);
        if (end != arg.length())
            throw std::exception{};
        filter.set(enyx::hw::index{acc_id});
    } catch (std::exception const&) {
        filter.set(enyx::hw::name{arg});
    }

    auto accelerators = enyx::hw::enumerate_accelerators(filter);
    if (accelerators.size() < 1) {
        std::ostringstream oss;
        oss << "No such accelerator: '" << arg << "'";
        throw std::runtime_error{oss.str()};
    }

    return accelerators.front();
}

static auto
load_model_conf(std::string const & filename)
{
    try {
        return enyx::fix::model::load_yaml_model(filename);
    } catch (std::exception & e) {
        std::clog << "Error loading FIX model: " << e.what() << std::endl;
        exit(EXIT_FAILURE);
    }
}

static auto
load_model_to_engine(std::string const & filename, std::string const & acc)
{
    auto acc_desc = find_accelerator(acc);
    enyx::fix::engine engine{acc_desc};

    auto model = load_model_conf(filename);
    engine.load_model(model);

    std::clog << "YAML FIX model loaded into FIX engine" << std::endl;
}

int main(int argc, char ** argv)
{
    auto args = argos::ArgumentParser("enyx-fix-model")
        .about("Fix model management")
        .add(argos::Option{"-a", "--accelerator"}.argument("accelerator")
             .help("The accelerator to use")
             .optional(true))
        .add(argos::Argument("command"))
        .add(argos::Argument("file"))
        .parse(argc, argv);

    auto command = args.value("command").as_string();
    auto file = args.value("file").as_string();

    if (command == "check") {
        load_model_conf(file);
        std::clog << "YAML FIX model valid" << std::endl;
    } else if (command == "load") {
        if (! args.has("--accelerator")) {
            std::cerr
                << "Accelerator option must be provided with the load command"
                << std::endl;
            std::exit(EXIT_FAILURE);
        }

        load_model_to_engine(file, args.value("--accelerator").as_string());
    }
}
