#pragma once

#include <ostream>
#include <map>

#include <stdexcept>
#include <enyx/fix/fields.hpp>
#include <enyx/fix/utils.hpp>

namespace enyx::fix {
/**
 * FIX Message used by the @ref Acceptor
 */
struct Message {
    /// @cond
    Message()
        : fields_{}
    {}

    Message(Message && other) = default;
    Message& operator=(Message && other) = default;

    Message& operator=(const Message & other) {
        for (auto & [tag, field]: other.fields_) {
            fields_[tag] = field;
        }
        return *this;
    }

    Message(const Message & other) {
        *this = other;
    }
    /// @endcond

    /**
     * Construct a message from a @p MsgType
     *
     * @param MsgType A fix @c MsgType<34>
     */
    Message(FieldTypes::String const & MsgType)
        : fields_{{field_tags::MsgType, MsgType}}
    {}

    template<typename FieldType>
    void
    getField(field_tags::Tag tag, FieldType & field) const
    {
        if (! getFieldIfSet(tag, field))
            throw std::runtime_error{"Field Not Found"};
    }

    template<typename FieldType>
    bool
    getFieldIfSet(field_tags::Tag tag, FieldType & field) const
    {
        if (fields_.count(tag) == 0)
            return false;
        auto const & value = fields_.at(tag);
        field = std::get<FieldType>(value);
        return true;
    }

    /**
     * Get a field using a @ref FieldValue
     *
     * @param value The @ref FieldValue to get
     * @throws std::runtime_error if the field is not set in this message
     */
    template<field_tags::Tag tag, typename FieldType>
    void
    getField(FieldValue<tag, FieldType> & value) const
    {
        getField(value.tag, value.field);
    }

    /**
     * Get a field if it is set
     *
     * @param value The @ref FieldValue to get
     * @return True if the field was found, false otherwise
     */
    template<field_tags::Tag tag, typename FieldType>
    bool
    getFieldIfSet(FieldValue<tag, FieldType> & value) const
    {
        return getFieldIfSet(value.tag, value.field);
    }

    /**
     * Set a field using a @ref FieldValue
     *
     * @param value The field to set
     */
    template<field_tags::Tag Tag, typename FieldType>
    void
    setField(FieldValue<Tag, FieldType> const & value)
    {
        setField(value.tag, value.field);
    }

    template<typename FieldType>
    void
    setField(field_tags::Tag tag, FieldType && field)
    {
        fields_[tag] = field;
    }

    template<typename FieldType>
    void
    setField(std::uint16_t tag, FieldType && field)
    {
        setField(static_cast<field_tags::Tag>(tag), field);
    }

    std::map<std::uint16_t, field> fields_;
};

template<char... MsgT>
struct AdminMsg : Message {
    static constexpr char MsgType_[] = {MsgT..., 0};

    static std::string
    MsgType() noexcept {
        return std::string(MsgType_, strlen(MsgType_));
    }
    AdminMsg()
        : Message{FieldTypes::String{MsgType()}}
    {}
};

using Heartbeat     = AdminMsg<'0'>;
using TestRequest   = AdminMsg<'1'>;
using ResendRequest = AdminMsg<'2'>;
using Reject        = AdminMsg<'3'>;
using SequenceReset = AdminMsg<'4'>;
using Logout        = AdminMsg<'5'>;
using Logon         = AdminMsg<'A'>;


std::ostream &operator<<(std::ostream & os, field const & field);

inline
std::ostream &operator<<(std::ostream & os, Message const &message)
{
    utils::joiner joiner{char{1}};
    for (auto const & [tag, field]: message.fields_) {
        os << joiner << tag << "=" << field;
    }
    return os;
}

}
