#pragma once

#include <optional>
#include <string>
#include <unordered_map>
#include <unordered_set>

#include <enyx/cores/types.hpp>
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw/c2a_stream.hpp>
#include <enyx/cores/data_stream/hw_source.hpp>
#include <enyx/cores/data_stream/hw_sink.hpp>
#include <enyx/cores/tcp/tcp.hpp>
#include <enyx/cores/tcp/context.hpp>
#include <enyx/cores/tcp/data_stream.hpp>

#include <enyx/fix/exceptions.hpp>
#include <enyx/fix/fields.hpp>
#include <enyx/fix/reject.hpp>
#include <enyx/fix/session.hpp>
#include <enyx/fix/tcp.hpp>
#include <enyx/fix/transport.hpp>

namespace enyx {
namespace fix {

/**
 * Should be thrown by user during @ref Application::onLogon to reject the
 * Initiator Logon<A> message if the user/password provided is deemed invalid.
 */
class RejectLogon: public FixError {
public:
    /**
     * Construct a RejectLogon error.
     *
     * @note @p error will be used as @c Text<58> in the outgoing Logout
     * message.
     */
    RejectLogon(std::string const & error)
        : FixError(error)
    {}
};

/**
 * @brief Application to be implemented by the user
 *
 * Provides the CompID used by the @ref Acceptor an various hooks.
 */
class Application {
public:
    /**
     * Create an application with a given @p senderCompID
     *
     * @param senderCompID The @c CompID used to identify this application.
     */
    Application(std::string const& senderCompID)
        : onLogon{}
        , onLogged{}
        , senderCompID_{senderCompID}
    {}

    /**
     * Called on each valid logon message, should be used to check credentials
     * if any.
     *
     * The whole message is passed as user/password are optional fields.
     * Example implementation:
     *
     * @code{.cpp}
     *     void
     *     onLogon(FixSourceHeader const& header,
     *             enyx::fix::Session const& session) {
     *
     *         enyx::fix::Username username;
     *         enyx::fix::Password password;
     *         if (! message.getFieldIfSet(username)) {
     *             throw enyx::fix::RejectLogon("No username specified");
     *         }
     *
     *         if (! message.getFieldIfSet(password)) {
     *             throw enyx::fix::RejectLogon("No password specified");
     *         }
     *     }
     * @endcode
     *
     * When this method call throws no error, logon process continues
     * and logon response message will be sent.
     * Use onLogged call to detect when logon response was sent.
     *
     * If the @c Logon message is invalid and should generate a @c Logout in
     * response, @ref onLogon should throw a @ref RejectLogon error.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The Logon message that should be checked for
     * username/password
     *
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onLogon;

    /**
     * Called when logon response message was sent.
     *
     * This can be used to detect when session is active,
     * and business messages can be transmitted and received.
     *
     * @param header The source for this session.
     *
     */
    std::function<void(FixSourceHeader const& header)> onLogged;


    /**
     * Called when connection was closed.
     * This is triggered on TCP connection close,
     * and not when receiving FIX logout message.
     * This ensures that custom logic can be notified
     * even when session is abruptly closed
     *
     * @param fix_session_id the corresponding TCP session index
     *
     */
    std::function<void(std::uint32_t fix_session_id)> onConnectionClosed;

    /**
     * Called when a valid Heartbeat message is received
     *
     * No action from user is expected from this callback. It should be used
     * for logs and debug.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The Heartbeat message that was received
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onHeartbeat;

    /**
     * Called when a valid TestRequest message is received
     *
     * No action from user is expected from this callback. It should be used
     * for logs and debug.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The TestRequest message that was received
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onTestRequest;

    /**
     * Called when a valid ResendRequest message is received
     *
     * No action from user is expected from this callback. It should be used
     * for logs and debug.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The ResendRequest message that was received
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onResendRequest;

    /**
     * Called when a valid Reject message is received
     *
     * No action from user is expected from this callback. It should be used
     * for logs and debug.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The Reject message that was received
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onReject;

    /**
     * Called when a valid SequenceReset message is received
     *
     * No action from user is expected from this callback. It should be used
     * for logs and debug.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The SequenceReset message that was received
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onSequenceReset;

    /**
     * Called when a valid Logout message is received
     *
     * No action from user is expected from this callback. It should be used
     * for logs and debug.
     *
     * @param header The message header which indicates which tcp session the
     *               message was received for.
     * @param message The Logout message that was received
     */
    std::function<void(FixSourceHeader const& header,
                       Message const& message)> onLogout;

    /**
     * Provide the senderCompID of an user application.
     */
    enyx::fix::SenderCompID
    senderCompID()
    {
        return enyx::fix::SenderCompID{senderCompID_};
    }

    /**
     * Virtual destructor, should be overriden by children
     */
    virtual ~Application() {};
protected:
    std::string senderCompID_;
};

class Acceptor;

/**
 * FIX Acceptor class
 *
 * This class behaves as a FIX Acceptor that only handles administrative
 * messages (everything else is dropped).
 */
class Acceptor {
public:

    /**
     * Construct a FIX Acceptor
     *
     * @param app The client application to be used
     * @param tcp The tcp stack to use
     * @param data_a2c The data stream used to receive data from the hardware
     * @param data_c2a The data stream used to send data to the hardware
     */
    Acceptor(Application & app,
             enyx::tcp::tcp const & tcp,
             enyx::hw::a2c_stream && data_a2c,
             enyx::hw::c2a_stream && data_c2a);

    ///@cond
    // Not documented
    void
    on_fix_event(const std::uint8_t * data, std::uint32_t size,
                 void const * metadata);

    void
    operator()(FixSourceHeader source,
               Message message);

    void
    operator()(FixSourceHeader source,
               FixErrHeader error,
               std::optional<Message> message);
    ///@endcond

    /**
     * Poll the fix Acceptor
     *
     * This method should be called continuously in the program main loop,
     * it is responsible for reading from the hardware
     */
    void
    poll();

    /**
     * Access the TcpSession manager
     */
    TcpSessionManager &
    tcp_session_manager();

    /**
     * Add an expected peer to the acceptor
     *
     * Non expected peers will be rejected
     */
    void
    add_peer(tcp::session_parameters const & params,
             Session::PeerCompID const& peer_comp_id);

    /**
     * Drop a FIX connection.
     *
     * Mostly used internally.
     */
    void
    drop_connection(std::uint16_t fix_session_id);

    /**
     * Drop any FIX connection associated to a TCP session id.
     *
     * Mostly used internally.
     */
    void
    cleanup_tcp_session(std::uint16_t tcp_session_id);

private:
    void send(std::uint32_t fix_session_id, Message & message);

    std::optional<uint16_t>
    find_connection_id(std::string peer_comp_id);

    void
    reject(std::uint32_t fix_session_id, reject::Reason reason,
           std::optional<Message> msg_opt=std::nullopt, std::uint64_t field=0);

    void
    handle_compid_problem(std::uint32_t fix_session_id,
                          std::optional<Message> msg_opt);

    void
    handle_seqnum_gap(FixSourceHeader source, Message message);

private:
    void
    on_sequence_reset(FixSourceHeader source, Message message);
    void
    on_logon(FixSourceHeader source, Message message);
    void
    on_heartbeat(FixSourceHeader source, Message message);
    void
    on_test_request(FixSourceHeader source, Message message);
    void
    on_logout(FixSourceHeader source, Message message);
    void
    on_resend_request(FixSourceHeader source, Message message);
    void on_reject(FixSourceHeader source, Message message);

    void
    logout_close_or_reject(FixSourceHeader source,
                           reject::Reason reason,
                           std::optional<Message> msg_opt,
                           std::uint64_t field);

private:
    enyx::tcp::tcp tcp_;
    enyx::hw::core fix_engine_;
    std::unordered_map<Session::PeerCompID, Session> sessions_;
    std::unordered_map<std::uint16_t, Connection> connections_;
    Application & app_;
    TcpSessionManager session_manager_;
    LowLevelTransport transport_;

    std::vector<TcpSessionToClose> to_close_;
};

}
}
