#pragma once
#include <cinttypes>
#include <cmath>
#include <cstdint>
#include <cstring>
#include <ctime>
#include <iomanip>
#include <sstream>
#include <string>
#include <vector>
#include <iostream>

#include <enyx/fix/fix_common.hpp>

namespace enyx::fix::FieldTypes {

namespace {
template<typename T>
inline
T
from_bytes(std::uint8_t const * data, std::size_t size)
{
    T ret;
    if (size < sizeof(ret)) {
        std::ostringstream err;
        err << "Could not deserialize field value: Data too small ("
            << size << " < " << sizeof(ret) << ")";
        throw std::runtime_error{err.str()};
    }
    std::memcpy(&ret, data, sizeof(ret));
    return ret;
}

}

#define LITTERAL_TYPE(type, lit) \
    struct type { \
        type () \
            : type(0) \
        {} \
        type (lit data) \
            : data_(data) \
        {} \
        type (std::uint8_t const * data, std::size_t size) \
            : type (from_bytes<lit>(data, size)) \
        {} \
        lit \
        value() const noexcept { \
            return data_; \
        } \
        std::string \
        str() const noexcept \
        { \
            return std::to_string(value()); \
        } \
        lit data_; \
    }

/**
 * @struct enyx::fix::FieldTypes::Char
 * @brief A field containing a single char
 *
 * @fn Char(char data)
 * @brief Construct a @ref Char from @p data
 * @param data A @c char
 * @public @memberof enyx::fix::FieldTypes::Char
 *
 * @fn char Char::value()
 * @brief Get the contained value
 * @return The value as a @c char
 * @public @memberof enyx::fix::FieldTypes::Char
 *
 * @fn std::string Char::str()
 * @brief Get the contained value as a @c std::string
 * @return The value as a @c std::string
 * @public @memberof enyx::fix::FieldTypes::Char
 */

LITTERAL_TYPE(Char, char);

/**
 * A field containing a boolean
 */
struct Boolean {
    /**
     * Construct a @ref Boolean from a @c char
     *
     * @param data Either @c 'Y' or @c 'N'
     *
     * @note Values other than @c 'Y' or @c 'N' are interpreted as @c 'N' but
     * @ref str() will return the provided value.
     */
    Boolean(char data)
        : data_(data)
    {}

    Boolean()
        : data_('N')
    {}

    /**
     * Construct a @ref Boolean from a @c bool
     *
     * @param data The data as a @c bool
     */
    Boolean(bool data)
        : data_(data ? 'Y' : 'N')
    {}

    Boolean(std::uint8_t const * data, std::size_t size)
        : Boolean(from_bytes<char>(data, size))
    {}

    /**
     * Get the contained value as a @c bool
     *
     * @return A @c bool
     */
    bool
    value() const noexcept {
        return data_ == 'Y';
    };

    /**
     * Get the contained value as a @c std::string
     *
     * @return The contained value
     */
    std::string
    str() const noexcept
    {
        return std::string{data_};
    }

    char data_;
};

struct FixedPointUDecimal {
    FixedPointUDecimal(std::uint64_t data)
        : value(data >> 4ULL)
        , scale(data & 0b1111ULL)
    {}

    explicit operator double() const
    {
        return value * std::pow(10, -scale);
    };

    explicit operator std::uint64_t() const
    {
        return value << 4ULL & scale;
    }

    std::uint64_t value;
    std::uint8_t scale;
};

struct FixedPointDecimal {
    static
    std::int64_t
    _upscale(std::uint64_t i60b)
    {
        std::uint64_t const first_bit = i60b & (1ULL << 59ULL);
        std::uint64_t value = i60b;
        if (first_bit)
            value |= 0xFULL << 60ULL;
        return static_cast<std::int64_t>(value);
    }

    FixedPointDecimal(std::uint64_t data)
        : value(_upscale(data >> 4ULL))
        , scale(data & 0xFULL)
    {}

    explicit operator double() const
    {
        return value * std::pow(10, -scale);
    }

    explicit operator std::uint64_t() const
    {
        return static_cast<std::uint64_t>(value) << 4ULL & scale;
    }

    std::int64_t value;
    std::uint8_t scale;
};

template<typename T>
struct AnyDecimal {
    AnyDecimal (std::uint64_t data)
        : data_(data)
    {}
    AnyDecimal(std::uint8_t const * data, std::size_t size)
        : AnyDecimal(from_bytes<std::uint64_t>(data, size))
    {}
    double
    value() const noexcept
    {
        return double(data_);
    };

    explicit operator std::uint64_t() const
    {
        return std::uint64_t{data_};
    }

    std::string
    str() const noexcept
    {
        return std::to_string(value());
    }
    T data_;
};

/**
 * @struct enyx::fix::FieldTypes::UDecimal
 * @brief A field containing an unsigned fixed point decimal
 *
 * @fn UDecimal(std::uint64_t data)
 * @brief Construct a @ref UDecimal from @p data
 * @param data A @c std::uint64_t representing an unsigned decimal
 * @public @memberof enyx::fix::FieldTypes::UDecimal
 *
 * @fn double UDecimal::value()
 * @brief Get the contained value
 * @return The value as a @c double
 * @public @memberof enyx::fix::FieldTypes::UDecimal
 *
 * @fn std::string UDecimal::str()
 * @brief Get the contained value as a @c std::string
 * @return The value as a @c std::string
 * @public @memberof enyx::fix::FieldTypes::UDecimal
 */
using UDecimal = AnyDecimal<FixedPointUDecimal>;


/**
 * @struct enyx::fix::FieldTypes::Decimal
 * @brief A field containing an signed fixed point decimal
 *
 * @fn Decimal(std::uint64_t data)
 * @brief Construct a @ref Decimal from @p data
 * @param data A @c std::uint64_t representing a signed decimal
 * @public @memberof enyx::fix::FieldTypes::Decimal
 *
 * @fn double Decimal::value()
 * @brief Get the contained value
 * @return The value as a @c double
 * @public @memberof enyx::fix::FieldTypes::Decimal
 *
 * @fn std::string Decimal::str()
 * @brief Get the contained value as a @c std::string
 * @return The value as a @c std::string
 * @public @memberof enyx::fix::FieldTypes::Decimal
 */
using Decimal = AnyDecimal<FixedPointDecimal>;


/**
 * @struct enyx::fix::FieldTypes::Int
 * @brief A field containing a signed integer
 *
 * @fn Int(std::int64_t data)
 * @brief Construct a @ref Int from @p data
 * @param data A @c std::int64_t
 * @public @memberof enyx::fix::FieldTypes::Int
 *
 * @fn std::int64_t Int::value()
 * @brief Get the contained value
 * @return The value as a @c std::int64_t
 * @public @memberof enyx::fix::FieldTypes::Int
 *
 * @fn std::string Int::str()
 * @brief Get the contained value as a @c std::string
 * @return The value as a @c std::string
 * @public @memberof enyx::fix::FieldTypes::Int
 */
LITTERAL_TYPE(Int, std::int64_t);

/**
 * @struct enyx::fix::FieldTypes::UInt
 * @brief A field containing an unsigned integer
 *
 * @fn UInt(std::uint64_t data)
 * @brief Construct a @ref UInt from @p data
 * @param data A @c std::uint64_t
 * @public @memberof enyx::fix::FieldTypes::UInt
 *
 * @fn std::uint64_t UInt::value()
 * @brief Get the contained value
 * @return The value as a @c std::uint64_t
 * @public @memberof enyx::fix::FieldTypes::UInt
 *
 * @fn std::string UInt::str()
 * @brief Get the contained value as a @c std::string
 * @return The value as a @c std::string
 * @public @memberof enyx::fix::FieldTypes::UInt
 */
LITTERAL_TYPE(UInt, std::uint64_t);

/**
 * @struct enyx::fix::FieldTypes::Currency
 * @brief A field containing a Currency
 *
 * @fn Currency(std::uint16_t data)
 * @brief Construct a @ref Currency from @p data
 * @param data A @c Currency code
 * @public @memberof enyx::fix::FieldTypes::Currency
 *
 * @fn std::uint16_t Currency::value()
 * @brief Get the contained value
 * @return The value as a @c std::uint16_t
 * @public @memberof enyx::fix::FieldTypes::Currency
 *
 * @fn std::string Currency::str()
 * @brief Get the contained value as a @c std::string
 * @return The value as a @c std::string
 * @public @memberof enyx::fix::FieldTypes::Currency
 */
LITTERAL_TYPE(Currency, std::uint16_t);

/**
 * A field containing a String
 */
struct String {
    String ()
        : data_{}
    {}

    /**
     * Construct a @ref String from a @c std::string
     *
     * @param data A @c String
     */
    String(std::string const & data)
        : data_(data)
    {}

    static char const *
    from_bytes(std::uint8_t const * data, std::size_t size)
    {
        char const * out = reinterpret_cast<char const *>(data);
        return out;
    }

    String(std::uint8_t const * data, std::size_t size)
        : data_(from_bytes(data, size))
    {}

    /**
     * Construct a @ref String from a @a C-style string
     *
     * @param data A @a C-style String
     */
    String(char * data)
        : data_(data)
    {}

    /**
     * Get the value
     * @return The value
     */
    std::string
    value() const noexcept
    {
        return data_;
    }

    /**
     * Get the value
     * @return the value
     *
     * @note This function is a duplicate of @ref value() but exits for
     * consistency with other field types.
     */
    std::string
    str() const noexcept
    {
        return value();
    }

    std::string data_;
};

/**
 * A year-month-day date
 */
struct [[gnu::packed]] Date {
    /// The day of month
    std::uint8_t day;
    /// The month of year
    std::uint8_t month;
    /// The year
    std::uint16_t year;
};

/**
 * A field containing a @ref Date
 */
struct UTCDateOnly {
    static Date
    from_uint(std::uint32_t data)
    {
        Date ret;
        std::memcpy(&ret, &data, sizeof(ret));
        return ret;
    }

    UTCDateOnly()
        : UTCDateOnly(0)
    {}

    /**
     * Construct a @ref UTCDateOnly from a raw data
     *
     * @param data The raw data
     */
    UTCDateOnly(std::uint32_t data)
        : data_(from_uint(data))
    {}

    UTCDateOnly(std::uint8_t const * data, std::size_t size)
        : UTCDateOnly(from_bytes<std::uint32_t>(data, size))
    {}

    /**
     * Get The value as a @ref Date
     *
     * @return The value as a @ref Date
     */
    Date
    value() const noexcept
    {
        return data_;
    }

    /**
     * Get the value as a @c std::string
     *
     * @return The value as a FIX String (YYYYMMDD)
     */
    std::string
    str() const noexcept {
        std::ostringstream oss;
        oss << value().year
            << std::uint64_t(value().month)
            << std::uint64_t(value().day);
        return oss.str();
    }

    Date data_;
};

/**
 * A timestamp in microseconds since epoch
 */
struct [[gnu::packed]] Timestamp {
    std::uint64_t microseconds;
};

/**
 * A field containing a timestamp.
 * Value is stored internally using timespec structure
 */
struct UTCTimestamp {

    timespec
    from_micros(Timestamp data)
    {
        timespec res;
        res.tv_sec =  data.microseconds / enyx::fix::MICROSECONDS_IN_SECOND;
        res.tv_nsec = (data.microseconds % enyx::fix::MICROSECONDS_IN_SECOND) * enyx::fix::NANOS_IN_MICROSECOND;
        return res;
    }

    /**
     * Create a @ref UTCTimestamp with the current timestamp from system clock.
     */
    UTCTimestamp()
        : data_(now())
    {}

    /**
     * Create a @ref UTCTimestamp from a timestamp
     *
     * @param data A timestamp
     */
    UTCTimestamp(timespec data)
        : data_(data)
    {}

    UTCTimestamp(Timestamp data)
        : UTCTimestamp(from_micros(data))
    {}

    UTCTimestamp(std::time_t data)
        : data_{data, 0}
    {}

    UTCTimestamp(std::uint8_t const * data, std::size_t size)
        : UTCTimestamp(from_bytes<Timestamp>(data, size))
    {}

    /**
     * Get the value as a @c timespec
     *
     * @return The contained value
     */
    timespec
    value() const noexcept
    {
        return data_;
    }

    /**
     * Get the value as a @c YYYMMDD-HH:MM:SS.mmm formatted string
     *
     * @return A @c std::string
     *
     * @note If the convertion to string fails, the result will be @c
     * <invalid-time>
     */
    std::string
    str() const noexcept
    {
        timespec val = value();
        auto utcsec = std::gmtime(&val.tv_sec);
        if (utcsec == nullptr) {
            return "<invalid-time>";
        }
        std::ostringstream oss;
        oss << std::put_time(utcsec, "%Y%m%d-%H:%M:%S") << "."
            << std::setfill('0') << std::setw(9) << val.tv_nsec;
        return oss.str();
    }

    static timespec
    now()
    {
        timespec now;
        timespec_get(&now, TIME_UTC);
        return now;
    }

    timespec data_;
};

#undef DECIMAL_TYPE
#undef STRING_TYPE
#undef LITTERAL_TYPE

}
