#pragma once

#include <cstdint>
#include <cstring>
#include <iostream>
#include <stdexcept>
#include <string>
#include <utility>

#include <enyx/fix/protocol.hpp>

namespace enyx::fix {

/**
 * Audit message direction.
 *
 * @note This type can be used with a <tt>ostream& operator<<</tt> that
 * displays @c Incoming
 * on incoming messages and @c Outgoing on outgoing messages.
 */
enum class AuditDirection {
    /// Incoming message
    Incoming = 0,
    /// Outgoing message
    Outgoing = 1,
};

/**
 * Object used to monitor the audit channel
 *
 * Example instantiation:
 * @code{.cpp}
 *
 * namespace ef = enyx::fix;
 *
 * ef::Audit audit{[] (ef::AuditDirection const & dir,
 *                     std::uint64_t seq_id,
 *                     std::uint8_t * const data,
 *                     std::size_t size) {
 *    char const * msg_cstr = reinterpret_cast<char const *>(data);
 *    std::string message(msg_cstr, strnlen(msg_cstr, size));
 *    std::cout << "Received " << dir << " '" << message
 *              << "' with SeqID: " << seq_id << std::endl;
 * }};
 *
 * enyx::data_stream::hw_source source{audit_stream, audit};
 *
 * for(;;) {
 *     source.poll_once();
 * }
 *
 * @endcode
 *
 * @tparam Handler Any callback type with the following prototype:
 * @code{.cpp}
 * void Handler(enyx::fix::AuditDirection const& direction,
 *              std::uint64_t seq_id,
 *              std::uint8_t * const data,
 *              std::size_t size);
 *
 * @endcode
 */
template<typename Handler>
class Audit {
public:
    /**
     * Construct an Audit object
     *
     * @param handler Callable that will be called on each message
     */
    Audit(Handler && handler)
        : handler_(std::forward<Handler>(handler))
    {}

    ///@cond
    void
    operator()(std::uint8_t const * data,
               std::size_t size,
               void const * metadata)
    {
        AuditHeader header{};
        if (size < sizeof(header)) {
            std::runtime_error{"Bad Audit message"};
        }

        std::memcpy(&header, data, sizeof(header));
        data += sizeof(header);
        size -= sizeof(header);

        handler_(static_cast<AuditDirection>(header.direction),
                 header.seq_id,
                 data,
                 size);
    }
    ///@endcond

private:
    Handler handler_;
};

inline
std::ostream &operator<<(std::ostream & os, AuditDirection const & direction)
{
    switch(direction) {
    case AuditDirection::Incoming:
        os << "Incoming";
        break;
    case AuditDirection::Outgoing:
        os << "Outgoing";
        break;
    }
    return os;
}

}
