"""Variable types"""

from collections import OrderedDict
import re
from typing import Dict, Type, List
from datetime import datetime
from functools import reduce
from ipaddress import IPv4Address

from .register import Register
from .register import PageCount, PageID, PageReady
from .register import Argument, Done, Return

from .utils import chain_variables


class Variable:
    """Simple variable containing one register"""
    def __init__(self, register: Register):
        self.register = register

    @property
    def type(self):
        """type property"""
        return type(self).__name__

    @property
    def comment(self):
        """Variable description"""
        return self.register.comment

    @property
    def read(self):
        """Readability property"""
        return self.register.read

    @property
    def write(self):
        """Writability property"""
        return self.register.write


class Bit(Variable):
    """Bit variable"""


class Hex(Variable):
    """Hex variable"""


class Int(Variable):
    """Int variable"""


class Enum(Variable):
    """Enum variable"""
    def __init__(self, register: Register, values: Dict[str, int]):
        super().__init__(register)
        self.values = OrderedDict(values)

    @property
    def name(self):
        """name property"""
        return self.register.name


class Bool(Variable):
    """Bool variable"""


class IPAddress(Variable):
    """IP address variable"""
    @staticmethod
    def ip_str_to_int(ipv4: str):
        """Get int value of a string IP address"""
        return int(IPv4Address(ipv4))


class MACAddress(Variable):
    """MAC address variable"""
    @staticmethod
    def mac_str_to_int(mac: str):
        """Helper to parse a string MAC address to an int."""
        return reduce(lambda x, y: (x << 8) + y,
                      map(lambda s: int(s, 16),
                          mac.split(':')))


class Time(Variable):
    """Time variable"""
    @staticmethod
    def time_str_fr_to_timestamp(time: str):
        """Get a timestamp from a dump formatted time"""
        return int(datetime.strptime(time, "%d/%m/%Y %H:%M:%S").timestamp())


class VariableContainer:
    """Complex variable containing sub-variables"""
    def __init__(self, name: str):
        self.name_ = name

    @property
    def name(self):
        """name property"""
        return self.name_.lower()

    @property
    def type(self):
        """type property"""
        return type(self).__name__


class String(VariableContainer):
    """String variable"""
    def __init__(self, name: str, index: Int, value: Int):
        super().__init__(name)
        self.index = index
        self.value = value

    @classmethod
    def from_registers(cls: Type, index: Int, value: Int):
        """Deduce name from index register and build String variable"""
        name = re.sub(r'_index$', '', index.register.name.lower())
        return cls(name, index, value)

    @property
    def comment(self):
        """Variable description"""
        return self.value.comment

    @property
    def variables(self):
        """List of variables"""
        return (self.index, self.value)


class Page(VariableContainer):
    """Page"""
    def __init__(self, name: str, count: PageCount,
                 page_id: PageID, ready: PageReady, variables: list,
                 values: list = None):
        # pylint: disable=too-many-arguments
        super().__init__(name)
        self.count = count
        self.page_id = page_id
        self.ready = ready
        self.variables_ = variables
        self.page_values = values if values is not None else []

    @property
    def name(self):
        """name property"""
        return self.page_id.register.name[:-3].lower()

    @property
    def variables(self):
        """List sub variables of page"""
        return tuple([self.count, self.page_id, self.ready]
                     + list(chain_variables(self.variables_)))

    @property
    def enums(self):
        """List enums of Page"""
        return tuple(filter(lambda v: isinstance(v, Enum), self.variables))


class Command(VariableContainer):
    """HW command"""
    def __init__(self, name: str, execute: Int,
                 args: List[Argument] = None, ret: Return = None,
                 done: Done = None, comment: str = ''):
        # pylint: disable=too-many-arguments
        super().__init__(name)
        self.execute = execute
        self.args = list(args) if args else []
        self.ret = list(ret) if ret else []
        self.done = done
        self.comment = comment

    @property
    def variables(self):
        """ Args + Rets (Dones ignored) """
        return tuple([self.execute] + list(self.args) + list(self.ret))


class PageValues(dict):
    """Define a specific type for page values dict"""
