/*
 * HFP rx module.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/module.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_char_device.h"
#include "enyx_hfp_bus.h"
#include "enyx_hfp_rx_usr.h"

static struct enyx_hfp_char_device_region * enyx_hfp_rx_char_device_region;

static ushort buffers_size_shift = 21; /* 2 MiB */
module_param(buffers_size_shift, ushort, S_IRUSR | S_IRGRP | S_IROTH);
MODULE_PARM_DESC(buffers_size_shift, "Buffer size shift (2^X)");

#define HFP_RX_ID 262
#define HFP_RX_MAJOR_0 0
#define HFP_RX_MAJOR_1 1

static DEFINE_HFP_DEVICE_TABLE(enyx_hfp_rx_device_table) = {
    {HFP_DEVICE_ID_MAJOR(HFP_RX_ID, HFP_RX_MAJOR_0)},
    {HFP_DEVICE_ID_MAJOR(HFP_RX_ID, HFP_RX_MAJOR_1)},
    {0},
};
MODULE_ALIAS_HFP_ID_MAJOR(HFP_RX_ID, HFP_RX_MAJOR_0);
MODULE_ALIAS_HFP_ID_MAJOR(HFP_RX_ID, HFP_RX_MAJOR_1);

struct enyx_hfp_drvdata
{
    struct enyx_hfp_rx_usr * rx;
};

static int
enyx_hfp_device_probe(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_drvdata * drvdata;
    int err;

    /* Bind only to tx under an autonomous splitter */
    if (enyx_hfp_get_device_parent(dev)->id.hardware_id != 17) {
        err = -ENODEV;
        goto err_parent_check;
    }

    drvdata = kzalloc(sizeof(*drvdata), GFP_KERNEL);
    if (! drvdata) {
        err = -ENOMEM;
        dev_err(&dev->device, "Can't allocate enyx_hfp rx struct");
        goto err_drvdata_alloc;
    }
    enyx_hfp_set_drvdata(dev, drvdata);

    drvdata->rx = create_enyx_hfp_rx_usr(dev,
                                    buffers_size_shift,
                                    enyx_hfp_rx_char_device_region);
    if (! drvdata->rx) {
        err = -ENODEV;
        goto err_create_enyx_hfp_rx_usr;
    }

    return 0;

err_create_enyx_hfp_rx_usr:
    kfree(drvdata);
err_drvdata_alloc:
err_parent_check:
    return err;
}

static void
enyx_hfp_device_remove(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_drvdata * drvdata = enyx_hfp_get_drvdata(dev);

    destroy_enyx_hfp_rx_usr(drvdata->rx);

    kfree(drvdata);
}

static struct enyx_hfp_driver enyx_hfp_rx_driver = {
    .name = THIS_MODULE->name,
    .id_table = enyx_hfp_rx_device_table,
    .probe = enyx_hfp_device_probe,
    .remove = enyx_hfp_device_remove,
};

static int __init
init_this_module(void)
{
    int err;

    pr_info("%s %s <support@enyx.com>\n",
            MODULE_NAME, ENYX_HFP_MODULES_VERSION);

    enyx_hfp_rx_char_device_region = enyx_hfp_char_device_region_create(MODULE_NAME);
    if (! enyx_hfp_rx_char_device_region) {
        err = -EINVAL;
        goto err_alloc_chrdev_region;
    }

    err = enyx_hfp_rx_class_register();
    if (err < 0)
        goto err_enyx_hfp_rx_class_register;

    err = enyx_hfp_register_driver(&enyx_hfp_rx_driver);
    if (err < 0)
        goto err_enyx_hfp_register_driver;

    return 0;

err_enyx_hfp_register_driver:
    enyx_hfp_rx_class_unregister();
err_enyx_hfp_rx_class_register:
    enyx_hfp_char_device_region_destroy(enyx_hfp_rx_char_device_region);
err_alloc_chrdev_region:
    return err;
}

module_init(init_this_module);

static void __exit
exit_this_module(void)
{
    enyx_hfp_unregister_driver(&enyx_hfp_rx_driver);

    enyx_hfp_rx_class_unregister();

    enyx_hfp_char_device_region_destroy(enyx_hfp_rx_char_device_region);

    pr_debug("%s Exited\n", MODULE_NAME);
}

module_exit(exit_this_module);

MODULE_AUTHOR("David Keller <david.keller@enyx.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Hfp rx module");
MODULE_VERSION(ENYX_HFP_MODULES_VERSION);

