/*
 * HFP net module.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/module.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_bus.h"
#include "enyx_hfp_net_dev.h"
#include "enyx_hfp_id_usr.h"

static bool is_igmp_only;
module_param(is_igmp_only, bool, S_IRUSR | S_IRGRP | S_IROTH);
MODULE_PARM_DESC(is_igmp_only, "Discard on TX any protocol but IGMP");

#define HFP_NET_ID 259
#define HFP_NET_MAJOR_0 0

static DEFINE_HFP_DEVICE_TABLE(enyx_hfp_net_device_table) = {
    {HFP_DEVICE_ID_MAJOR(HFP_NET_ID, HFP_NET_MAJOR_0)},
    {0},
};
MODULE_ALIAS_HFP_ID_MAJOR(HFP_NET_ID, HFP_NET_MAJOR_0);

struct enyx_hfp_drvdata
{
    struct enyx_hfp_net_dev * net;
    struct enyx_hfp_id_usr * id;
};

static int
enyx_hfp_device_probe(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_drvdata * drvdata;
    int err;

    drvdata = kzalloc(sizeof(*drvdata), GFP_KERNEL);
    if (! drvdata) {
        err = -ENOMEM;
        dev_err(&dev->device, "Can't allocate enyx_hfp net struct");
        goto err_enyx_hfp_net_drvdata_alloc;
    }
    enyx_hfp_set_drvdata(dev, drvdata);

    drvdata->id = enyx_hfp_id_usr_create(dev);
    if (! drvdata->id) {
        err = -ENODEV;
        goto err_create_enyx_hfp_id_usr;
    }

    drvdata->net = enyx_hfp_net_dev_create(dev, is_igmp_only);

    return 0;

err_create_enyx_hfp_id_usr:
    kfree(drvdata);
err_enyx_hfp_net_drvdata_alloc:
    return err;
}

static void
enyx_hfp_device_remove(struct enyx_hfp_device * dev)
{
    struct enyx_hfp_drvdata * drvdata = enyx_hfp_get_drvdata(dev);

    if (drvdata->net)
        enyx_hfp_net_dev_destroy(drvdata->net);

    enyx_hfp_id_usr_destroy(drvdata->id);

    kfree(drvdata);
}

static struct enyx_hfp_driver enyx_hfp_net_driver = {
    .name = THIS_MODULE->name,
    .id_table = enyx_hfp_net_device_table,
    .probe = enyx_hfp_device_probe,
    .remove = enyx_hfp_device_remove,
};

static int __init
init_this_module(void)
{
    int err;

    pr_info("%s %s <support@enyx.com>\n",
            MODULE_NAME, ENYX_HFP_MODULES_VERSION);

    err = enyx_hfp_register_driver(&enyx_hfp_net_driver);
    if (err < 0)
        goto err_enyx_hfp_register_driver;

    return 0;

err_enyx_hfp_register_driver:
    return err;
}

module_init(init_this_module);

static void __exit
exit_this_module(void)
{
    enyx_hfp_unregister_driver(&enyx_hfp_net_driver);

    pr_debug("%s Exited\n", MODULE_NAME);
}

module_exit(exit_this_module);

MODULE_AUTHOR("David Keller <david.keller@enyx.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Hfp net module");
MODULE_VERSION(ENYX_HFP_MODULES_VERSION);

