#pragma once

#include <stdlib.h>
#include <stdint.h>

#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/properties.h>

/**
 * Opaque class representing filter.
 *
 * This class can be used as a filter to find specific device(s)
 * or as a source of information on a discovered device(s).
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_filter enyx_hw_filter;

/**
 * Create an empty filter object
 *
 * @returns The created property object or NULL if an error
 *          occurred (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_filter *
enyx_hw_filter_create(void);

/**
 * Retrieve the UID property
 *
 * @param filter The filter object to query
 * @param buffer The buffer to fill with the uid
 * @param capacity The buffer capacity
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_get_uid(enyx_hw_filter const * filter, char * buffer, size_t capacity);

/**
 * Set the UID property
 *
 * @param filter The filter object to update
 * @param uid The uid to set
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_set_uid(enyx_hw_filter * filter, const char * uid);

/**
 * Destroy a filter object
 *
 * @param filter The filter instance to destroy
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_filter_destroy(enyx_hw_filter * filter);

/**
 * Retrieve the index property
 *
 * @param filter The filter object to query
 * @param index The integer to fill with the index
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_get_index(enyx_hw_filter const * filter, uint32_t * index);

/**
 * Set the index property
 *
 * @param filter The filter object to update
 * @param index The index to set
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_set_index(enyx_hw_filter * filter, uint32_t index);

/**
 * Retrieve the name property
 *
 * @param filter The filter object to query
 * @param buffer The buffer to fill with the name
 * @param capacity The buffer capacity
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_get_name(enyx_hw_filter const * filter, char * buffer, size_t capacity);

/**
 * Set the name property
 *
 * @param filter The filter object to update
 * @param name The name to set
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_set_name(enyx_hw_filter * filter, const char * name);

/**
 * Retrieve the mtu property
 *
 * @param filter The filter object to query
 * @param mtu The integer to fill with the mtu
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 * @deprecated replaced by a2c/c2a stream get_mtu methods
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_get_mtu(enyx_hw_filter const * filter, uint32_t * mtu)
    ENYX_HW_C_DEPRECATED("replaced by a2c/c2a stream get_mtu methods");

/**
 * Set the mtu property
 *
 * @param filter The filter object to update
 * @param mtu The mtu to set
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 * @deprecated replaced by a2c/c2a stream get_mtu methods
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_set_mtu(enyx_hw_filter * filter, uint32_t mtu)
    ENYX_HW_C_DEPRECATED("replaced by a2c/c2a stream set_mtu methods");

/**
 * Retrieve the size property
 *
 * @param filter The filter object to query
 * @param size The integer to fill with the size
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_get_size(enyx_hw_filter const * filter, size_t * size)
    ENYX_HW_C_DEPRECATED("replaced by a2c/c2a stream get_size methods");

/**
 * Set the size property
 *
 * @param filter The filter object to update
 * @param size The size to set
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_set_size(enyx_hw_filter * filter, size_t size)
    ENYX_HW_C_DEPRECATED("replaced by a2c/c2a stream set_size methods");

/**
 * Retrieve the driver property
 *
 * @param filter The filter object to query
 * @param driver The integer to fill with the driver
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.9.5
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_get_driver(enyx_hw_filter const * filter, int * driver);

/**
 * Set the driver property
 *
 * @param filter The filter object to update
 * @param driver The driver to set
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.9.5
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_set_driver(enyx_hw_filter * filter, int driver);

/**
 * Test if two filter object are equals.
 *
 * @param a First operand
 * @param b Second operand
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_filter_cmp(enyx_hw_filter const * a, enyx_hw_filter const * b);

