/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  core_descriptions.
 */
#pragma once

#include <enyx/hw_c/core.h>
#include <enyx/hw_c/core_description.h>
#include <enyx/hw_c/symbol_visibility.h>

/**
 * This class is a core_description container.
 *
 * It is able to map the correct core_description to
 * the correct core according to its version
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_core_descriptions enyx_hw_core_descriptions;

/**
 *  Build core_descriptions from a folder containing xml files or a single
 *  xml file.
 *
 *  @param path the path
 *  @return The built core descriptions on success, NULL on error (with @b
 *  errno set accordingly)
 *  Common errno values:
 *      - @b EINVAL if path is NULL
 *      - @b ENOENT if path does not exist
 *      - @b EINVAL if path exists but is neither a regular file nor a directory
 *      - @b EINVAL if an error occured during the parsing of xml files (check
 *        logs)
 *
 *  Check the manpage of @b scandir and @b stat, for more detailled
 *  explanations.
 *
 *  @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_descriptions *
enyx_hw_core_descriptions_from_xml(const char * path);

/**
 * Validate an xml file containing core descriptions using an XML Schema
 * Definition.
 *
 * @param xml_file The xml file to validate
 * @param xsd The schema to use
 * @return 0 on success, -1 on failure.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descriptions_validate_xml_file(char const * xml_file, char const * xsd);

/**
 * Build core_descriptions from scratch. core_descriptions are a collection
 * of cores.
 *
 * @return The built core descriptions on success, NULL on error (with @b
 * errno set accordingly).
 * Errno can only be @b ENOMEM.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_descriptions *
enyx_hw_core_descriptions_from_scratch(void);

/**
 * Add two sets of core_descriptions.
 *
 * The two sets are merged into @p first
 *
 * @param first The first operand (modified in place).
 * @param second The second operand (unmodified).
 * @return @p first on success, NULL on failure (with @b errno set accordingly).
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_descriptions *
enyx_hw_core_descriptions_merge(enyx_hw_core_descriptions * first,
                                enyx_hw_core_descriptions const * second);


/**
 * Add a @p core description to an existing core @p descriptions.
 *
 * @param descriptions The core descriptions to be modified.
 * @param core The new description. It is *not* usable after being passed as
 * arguments but must still be freed.
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 * Possible errno values:
 *     - @b EINVAL if either @p descriptions or @p core is NULL
 *     - @b EINVAL if a another core with the same hardware id, but a different
 *       name already exists in @p descriptions
 *     - @b EEXIST if the same core (hardware id/name/version) already exists.
 *     - @b ENOMEM if internal storage could not be allocated for the new core.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY
int enyx_hw_core_descriptions_add_core(enyx_hw_core_descriptions * descriptions,
                                       enyx_hw_core_description const* core);

/**
 * Destroy a core_descriptions object
 *
 * @param d The core descriptions to destroy
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_core_descriptions_destroy(enyx_hw_core_descriptions * d);

/**
 * Find the best core_description for the core @p descriptor
 * from core @p descriptions instance
 *
 * @param descriptor The core descriptor to use as key
 * @param descriptions The core_description instance to search
 * @return the core description on success, NULL on error
 *         (@b errno is set accordingly)
 * Possible errno values:
 *     - @b EINVAL if @p descriptions or @p core is NULL
 *     - @b ESRCH if no such core was found.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_description *
enyx_hw_core_descriptions_find_by_desc(enyx_hw_core_descriptions const * descriptions,
                                       enyx_hw_core_descriptor const * descriptor);

/**
 * Find the best core_description for the core @p core
 * from core @p descriptions instance
 *
 * @param core The core requesting a description
 * @param descriptions The core_description instance to earcj
 * @return the core description on success, NULL on error
 *         (@b errno is set accordingly)
 * Possible errno values:
 *     - @b EINVAL if @p descriptions or @p core is NULL
 *     - @b ESRCH if no such core was found.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_description *
enyx_hw_core_descriptions_find(enyx_hw_core_descriptions const * descriptions,
                               enyx_hw_core const * core);

/**
 * Get the @p hardware_id of a core from its @p name using core
 * @p descriptions.
 *
 * @param descriptions The descriptions to query.
 * @param name The name of the core.
 * @param[out] hardware_id the returned hardware id.
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 * Possible errno values:
 *      - @b EINVAL if @p descriptions, @p name or @p hardware_id is NULL.
 *      - @b ESRCH if no core with such name was found.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descriptions_get_hardware_id(enyx_hw_core_descriptions const * descriptions,
                                          char const * name,
                                          uint16_t * hardware_id);

/**
 * Get the name of a core from  its @p hardware_id and core @p descriptions.
 *
 * @param descriptions The descriptions to query.
 * @param hardware_id The hardware id
 * @param[out] buffer Output buffer for the name.
 * @param size Size of @p buffer.
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 * Possible errno values:
 *      - @b EINVAL if @p descriptions, @p name or @p hardware_id is NULL.
 *      - @b ESRCH if no core with such hardware_id was found.
 *      - @b ENOBUFS if buffer is too small.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_core_descriptions_get_name(enyx_hw_core_descriptions const * descriptions,
                                   uint16_t hardware_id,
                                   char * buffer,
                                   size_t size);
