/**
 * @file
 *
 * Contains the types and functions related to the
 * CPU to Accelerator data streaming.
 */
#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/hw_c/properties.h>
#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/fwd.h>

/**
 * This class references a c2a_stream
 *
 * It is used to construct a c2a_stream, but doesn't
 * constitute ownership of it.
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_c2a_stream_descriptor enyx_hw_c2a_stream_descriptor;

/**
 * C2A stream flag to be sent along with the data.
 *
 * @note The warmup flag is only available on firmwares with HFP > 1.1.0
 *
 * @since 5.11.0
 */
typedef enum {
    ENYX_HW_C2A_STREAM_FLAG_NONE   = 0,
    ENYX_HW_C2A_STREAM_FLAG_WARMUP = 1 << 0,
    // ENYX_HW_C2A_STREAM_FLAG_LAST = 1 << 15,
} enyx_hw_c2a_stream_flags;

/**
 * Retrieve the properties associated with a c2a stream @p descriptor
 *
 * @param descriptor The c2a stream descriptor object to query
 * @returns A pointer to the properties object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_properties const *
enyx_hw_c2a_stream_descriptor_get_properties(enyx_hw_c2a_stream_descriptor const * descriptor);

/**
 * CPU to Accelerator stream configuration
 *
 * @since 5.0.0
 */
typedef struct
{
} enyx_hw_c2a_stream_configuration;

/**
 * Create a CPU to Accelerator channel from an @p descriptor and a @p c
 *
 * @param descriptor The stream descriptor
 * @param c The stream configuration
 * @returns The created stream object or NULL if an error
 *          occurred (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_c2a_stream *
enyx_hw_c2a_stream_create(enyx_hw_c2a_stream_descriptor const* descriptor,
                          enyx_hw_c2a_stream_configuration const* c);

/**
 * Detroy a stream object @p c
 *
 * @param s The stream to destroy
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_c2a_stream_destroy(enyx_hw_c2a_stream * s);

/**
 * Retrieve the accelerator associated with a c2a_stream descriptor @p d
 *
 * @param d The descriptor to query
 * @returns A pointer to the accelerator object
 *
 * @since 5.5.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_accelerator *
enyx_hw_c2a_stream_get_accelerator(enyx_hw_c2a_stream_descriptor const * d);

/**
 * Retrieve the descriptor associated with a c2a stream @p c
 *
 * @param s The c2a stream object to query
 * @returns A pointer to the descriptor object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_c2a_stream_descriptor const *
enyx_hw_c2a_stream_get_descriptor(enyx_hw_c2a_stream const * s);

/**
 * Free a previously getted descriptor.
 *
 * @param descriptor The descriptor to free.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_c2a_stream_put_descriptor(enyx_hw_c2a_stream_descriptor const * descriptor);

/**
 * Retrieve the stream mtu
 *
 * @param s The c2a stream object to query
 * @param mtu The integer to fill with the mtu
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.6.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_get_mtu(enyx_hw_c2a_stream const * s, uint32_t * mtu);

/**
 * Retrieve the stream size
 *
 * @param s The c2a stream object to query
 * @param size The integer to fill with the size
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.11.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_get_size(enyx_hw_c2a_stream const * s, size_t * size);

/**
 * Send @p size byte(s) from @p data to the accelerator.
 *
 * @param s The c2a stream object to use
 * @param data Pointer to the data to send.
 * @param size The byte(s) count to send.
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @note When the accelerator can't keep up with the bandwidth, @b errno
 *       will be set to @b EAGAIN. In that case, you may call
 *       send again.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_send(enyx_hw_c2a_stream * s,
                        void const * data, uint32_t size);

/**
 * Send @p size byte(s) from @p data to the accelerator.
 *
 * @param s The c2a stream object to use
 * @param data Pointer to the data to send.
 * @param size The byte(s) count to send.
 * @param flags The flags to set on data packet.
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @note When the accelerator can't keep up with the bandwidth, @b errno
 *       will be set to @b EAGAIN. In that case, you may call
 *       send again.
 *
 * @since 5.11.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_send2(enyx_hw_c2a_stream * s,
                         void const * data, uint32_t size,
                         enyx_hw_c2a_stream_flags flags);

/**
 * Describe a buffer to send.
 *
 * @see send()
 *
 * @since 5.0.0
 */
typedef struct
{
    /// Pointer to the data to send.
    void const * data_ptr;

    /// Byte(s) size of the data to send.
    uint32_t data_size;
} enyx_hw_c2a_stream_io_vec;

/**
 * Send @p size byte(s) from @p data to the accelerator.
 *
 * @param s The c2a stream object to use
 * @param vectors Pointer to an array of @b io vectors describing
 *        the data to send.
 * @param vector_count The @p vectors count.
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @note When the accelerator can't keep up with the bandwidth, @b errno
 *       will be set to @b EAGAIN. In that case, you may call
 *       send again.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_send_io_vec(enyx_hw_c2a_stream * s,
                               enyx_hw_c2a_stream_io_vec const * vectors,
                               uint32_t vector_count);

/**
 * Send @p size byte(s) from @p data to the accelerator.
 *
 * @param s The c2a stream object to use
 * @param vectors Pointer to an array of @b io vectors describing
 *        the data to send.
 * @param vector_count The @p vectors count.
 * @param flags The flags to set on data packet.
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @note When the accelerator can't keep up with the bandwidth, @b errno
 *       will be set to @b EAGAIN. In that case, you may call
 *       send again.
 *
 * @since 5.11.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_send_io_vec2(enyx_hw_c2a_stream * s,
                                enyx_hw_c2a_stream_io_vec const * vectors,
                                uint32_t vector_count,
                                enyx_hw_c2a_stream_flags flags);


/**
 * Check if @p s is ready to send data
 *
 * @param s The stream to check.
 * @return true if @p s is ready to send data.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY bool
enyx_hw_c2a_stream_is_ready(enyx_hw_c2a_stream const *s);

/**
 * Check if @p s is currently processing data
 *
 * @param s The stream to check.
 * @return true if the internal @p s queue is empty, false otherwise.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY bool
enyx_hw_c2a_stream_is_idle(enyx_hw_c2a_stream const *s);

/**
 * Wait for @p c2a device to be idle.
 *
 * @param s The tx device instance to wait for.
 * @param us_timeout The timeout until the function returns
 * @returns A positive number on success otherwise @b 0
 *        shall be returned and @b errno will be set to
 *        indicate the error.
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_wait_until_idle(enyx_hw_c2a_stream const *s,
                                   uint32_t us_timeout);

/**
 * Retrieve the stream mtu
 *
 * @param descriptor the c2a stream descriptor to query
 * @param mtu the integer to fill with the mtu
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.11.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_descriptor_get_mtu(enyx_hw_c2a_stream_descriptor const * descriptor,
                                      uint32_t * mtu);

/**
 * Retrieve the stream size
 *
 * @param descriptor the c2a stream descriptor to query
 * @param size the integer to fill with the size
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.11.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_c2a_stream_descriptor_get_size(enyx_hw_c2a_stream_descriptor const * descriptor,
                                       size_t * size);
