ENYX_HW_NAMESPACE_BEGIN

namespace mocking {

inline
mocked_register::mocked_register(::enyx_hwm_core * core,
                                 std::uint64_t byte_offset,
                                 std::uint8_t bit_offset,
                                 std::uint8_t bit_width)
    : mocked_register(core, {byte_offset, bit_offset, bit_width})
{}

inline
mocked_register::mocked_register(core_address_space * address_space,
                                 std::uint64_t byte_offset,
                                 std::uint8_t bit_offset,
                                 std::uint8_t bit_width)
    : mocked_register(address_space, {byte_offset, bit_offset, bit_width})
{}

inline
mocked_register::mocked_register(::enyx_hwm_core * core,
                                 register_description const & desc)
    : description(desc), handle_(::enyx_hwm_core_add_register(core, &desc),
                                 [] (::enyx_hwm_register *) {})
{
    if (handle_ == nullptr)
        throw std::system_error{errno, std::generic_category(),
                                "Failed to add register"};
    _set_callbacks();
}

inline
mocked_register::mocked_register(core_address_space * address_space,
                                 register_description const & desc)
    : description(desc), handle_(std::make_shared<::enyx_hwm_register>())
{
    if (handle_ == nullptr)
        throw std::system_error{errno, std::generic_category(),
                                "Failed to create register"};
    handle_->description = desc;
    handle_->core_address_space = address_space;
    _set_callbacks();
}

inline
result<void>
mocked_register::trigger_write() noexcept
{
    if (::enyx_hwm_register_trigger_write(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline
result<void>
mocked_register::trigger_read() noexcept
{
    if (::enyx_hwm_register_trigger_read(handle()) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline
std::uint64_t
mocked_register::get() const noexcept
{
    std::uint64_t value = 0;
    ::enyx_hwm_register_get(handle(), &value);
    return value;
}

inline
void
mocked_register::set(std::uint64_t new_value) noexcept
{
    ::enyx_hwm_register_set(handle(), new_value);
}

inline ::enyx_hwm_register *
mocked_register::handle() noexcept
{
    return handle_.get();
}

inline ::enyx_hwm_register const *
mocked_register::handle() const noexcept
{
    return handle_.get();
}

inline void
mocked_register::update_str(std::string const& string_value,
                            std::size_t chunk_index, std::size_t chunk_size,
                            mocked_register & value_reg)
{
    if (::enyx_hwm_register_update_str(string_value.data(),
                                              string_value.size(),
                                              chunk_index, chunk_size,
                                              value_reg.handle()) < 0)
        throw std::system_error{errno, std::generic_category(),
                                 "Could not update string register"};
}

inline
void
mocked_register::_set_callbacks()
{
    handle_->context = this;
    handle_->on_write = [] (void * context, uint64_t value) -> int {
        mocked_register * thiz = reinterpret_cast<mocked_register*>(context);
        if (thiz->on_write == nullptr) {
            return 0;
        }
        auto result = thiz->on_write(value);
        if (! result) {
            errno = result.error().value();
            return -1;
        }
        return 0;
    };
    handle_->on_read = [] (void * context, uint64_t * value) -> int {
        mocked_register * thiz = reinterpret_cast<mocked_register*>(context);
        if (thiz->on_read == nullptr) {
            *value = thiz->get();
            return 0;
        }
        auto result = thiz->on_read();
        if (! result) {
            errno = result.error().value();
            return -1;
        }
        *value = result.value();
        return 0;
    };
}

} /* namespace mocking */

ENYX_HW_NAMESPACE_END
