#pragma once

#include <cstdint>
#include <cstring>
#include <functional>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw_c/mocking/a2c_stream.h>

ENYX_HW_NAMESPACE_BEGIN
namespace mocking {

/**
 * Represent a mocked A2C stream interface.
 *
 * In order to customize it, please inherit and register it using the mocked
 * accelerator class.
 */
class a2c_stream_interface
{
public:
    a2c_stream_interface();
    /**
     * Destructor.
     */
    virtual ~a2c_stream_interface() = default;

    /**
     * Called when the stream is created.
     *
     * @return 0 on success, -1 on error with @b errno set.
     */
    std::function<int()> on_create;

    /**
     * Called when the stream is destroyed.
     */
    std::function<void()> on_destroy;

    /**
     * Called to get the uid.
     *
     * @return the uid
     */
    std::function<std::string()> get_uid;

    /**
     * Called to get the name.
     *
     * @return the name
     */
    std::function<std::string()> get_name;

    /**
     * Called to get the mtu.
     *
     * @return the mtu
     */
    std::function<std::uint32_t()> get_mtu;

    /**
     * Called to get the size.
     *
     * @return the size
     */
    std::function<std::size_t()> get_size;

    /**
     * Called to invoke @p on_data handler on received packet(s) of stream.
     *
     * @note The function should return 0 if no data is available
     *
     * @param on_data The handler to invoke on received packet.
     * @param opaque The opaque pointer to forward to the on_data handler.
     * @return The count of received packets.
     */
    std::function<size_t(enyx_hw_a2c_stream_handler on_data,
                        void * opaque)> poll_once;

    /**
     * Called to retrieve the total packet count received by stream.
     *
     * @return The packet count
     */
    std::function<std::uint32_t()> get_packet_count;

    /**
     * Called to retrieve the stream backpressure.
     *
     * @return the stream backpresuure
     */
    std::function<std::uint32_t()> get_backpressure;

    /**
     * Called to retrieve the total fifo errors received by stream.
     *
     * @return the fifo errors
     */
    std::function<std::uint32_t()> get_fifo_errors;

    /**
     * Called to retrieve the stream truncated packets.
     *
     * @return the stream backpresuure
     */
    std::function<std::uint32_t()> get_truncated_count;

    /**
     * Called to retrieve the total errors received by stream.
     *
     * @return the error count
     */
    std::function<std::uint32_t()> get_errors;

    /**
     * Called to retrieve the total usage received by stream.
     *
     * @return the usage
     */
    std::function<std::uint32_t()> get_usage;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_a2c_stream_interface *
    handle() noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_a2c_stream_interface const *
    handle() const noexcept;
private:
    std::unique_ptr<::enyx_hwm_a2c_stream_interface> handle_;
};

using basic_a2c_stream ENYX_HW_CXX_DEPRECATED(
            "Replaced by a2c_stream_interface") = a2c_stream_interface;

} /* namespace mocking */
ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/a2c_stream.ipp>
