/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator device.
 */
#pragma once

#include <enyx/hw_c/accelerator.h>

#include <cerrno>
#include <cstdint>
#include <system_error>
#include <vector>
#include <memory>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/fwd.hpp>
#include <enyx/hw/properties.hpp>
#include <enyx/hw/accelerator_descriptor.hpp>

#include <enyx/hw_c/accelerator.h>
#include <enyx/hw_c/mmio.h>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_accelerator
 */
class accelerator final
{
public:
    /**
     * Construct an accelerator from its @p descriptor
     *
     * @param descriptor The accelerator unique descriptor.
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    explicit
    accelerator(accelerator_descriptor const& descriptor);

    /**
     * Get a descriptor for this accelerator.
     *
     * @return the corresponding descriptor.
     *
     * @since 5.0.0
     */
    accelerator_descriptor
    get_descriptor() const noexcept;

    /**
     * Retrieve all the accelerator's mmios descriptors
     *
     * @return All the mmios found on this system.
     *
     * @since 5.0.0
     */
    std::vector<mmio_descriptor>
    enumerate_mmios() const noexcept;

    /**
     * Retrieve all the accelerator's mmios matching @p filter
     *
     * This function is used to request particular mmio(s).
     * The @p filter argument should be filled with properties
     * unique to the requeted mmio(s).
     *
     * @param filter mmio filter
     * @return All the mmio_descriptor with equivalent @p filter
     *         found on this accelerator.
     *
     * @since 5.0.0
     */
    std::vector<mmio_descriptor>
    enumerate_mmios(filter const& filter) const noexcept;

    /**
     * Retrieve all the accelerator's c2a_stream
     *
     * @return All the c2a_stream found on this system.
     *
     * @since 5.0.0
     */
    std::vector<c2a_stream_descriptor>
    enumerate_c2a_streams() const noexcept;

    /**
     * Retrieve all the accelerator's c2a_streams matching @p filter
     *
     * This function is used to request particular c2a_stream(s).
     * The @p filter argument should be filled with properties
     * unique to the requeted c2a_stream(s).
     *
     * @param filter c2a_stream filter
     * @return All the c2a_stream_descriptors with equivalent @p filter
     *         found on this accelerator.
     *
     * @since 5.0.0
     */
    std::vector<c2a_stream_descriptor>
    enumerate_c2a_streams(filter const& filter) const noexcept;

    /**
     * Retrieve all the accelerator's a2c_stream
     *
     * @return All the a2c_stream found on this system.
     *
     * @since 5.0.0
     */
    std::vector<a2c_stream_descriptor>
    enumerate_a2c_streams() const noexcept;

    /**
     * Retrieve all the accelerator's a2c_streams matching @p filter
     *
     * This function is used to request particular a2c_stream(s).
     * The @p filter argument should be filled with properties
     * unique to the requeted a2c_stream(s).
     *
     * @param filter a2c_stream filter
     * @return All the a2c_stream_descriptors with equivalent @p filter
     *         found on this accelerator.
     *
     * @since 5.0.0
     */
    std::vector<a2c_stream_descriptor>
    enumerate_a2c_streams(filter const& filter) const noexcept;

    /**
     * Direct access to the underlying C accelerator object.
     *
     * @return The C accelerator object.
     *
     * @since 5.0.0
     */
    enyx_hw_accelerator * handle() noexcept;

    /**
     * Direct access to the underlying C accelerator object.
     *
     * @return The C accelerator object.
     *
     * @since 5.0.0
     */
    enyx_hw_accelerator const * handle() const noexcept;

private:
    accelerator_descriptor descriptor_;
    std::shared_ptr<enyx_hw_accelerator> accelerator_;
};

/**
 * @copydoc enyx_hw_mmio_descriptor
 */
class mmio_descriptor final
{
public:
    /**
     * Retrieve the mmio's properties
     *
     * @return The mmio associated properties.
     *
     * @since 5.0.0
     */
    const properties
    get_properties() const;

    /**
     * Construct a descriptor from a C descriptor.
     *
     * @param owner The owner accelerator.
     * @param descriptor The C descriptor to wrap.
     * @throws std::system_error if descriptor is nullptr.
     *
     * @since 5.0.0
     */
    mmio_descriptor(accelerator const& owner,
                    enyx_hw_mmio_descriptor const * descriptor);

    /**
     * Direct access to the underlying C mmio_descriptor object.
     *
     * @return The C mmio_descriptor object.
     *
     * @since 5.0.0
     */
    enyx_hw_mmio_descriptor const * handle() const noexcept;

    /**
     * Retrieve the related accelerator.
     *
     * @return A reference to the accelerator.
     *
     * @since 5.5.0
     */
    accelerator const&
    get_accelerator() const noexcept;

private:
    accelerator accelerator_;
    std::shared_ptr<enyx_hw_mmio_descriptor const> descriptor_;
};

/**
 * @copydoc enyx_hw_a2c_stream_descriptor
 */
class a2c_stream_descriptor final
{
public:
    /**
     * Retrieve the a2c_stream's properties
     *
     * @return The a2c_stream associated properties.
     *
     * @since 5.0.0
     */
    const properties
    get_properties() const;

    /**
     * Construct a descriptor from a C descriptor.
     *
     * @param owner The owner accelerator
     * @param descriptor The C descriptor to wrap.
     * @throws std::system_error if descriptor is nullptr.
     *
     * @since 5.0.0
     */
    a2c_stream_descriptor(accelerator const& owner,
                          enyx_hw_a2c_stream_descriptor const * descriptor);

    /**
     * Direct access to the underlying C a2c_stream_descriptor object.
     *
     * @return The C a2c_stream_descriptor object.
     *
     * @since 5.0.0
     */
    enyx_hw_a2c_stream_descriptor const * handle() const noexcept;

    /**
     * Retrieve the related accelerator.
     *
     * @return A reference to the accelerator.
     *
     * @since 5.5.0
     */
    accelerator const&
    get_accelerator() const noexcept;

    /**
     * Retrieve the channel MTU
     *
     * @return A result object containing either the mtu or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t>
    get_mtu() const noexcept;

    /**
     * Retrieve the channel size
     *
     * @return A result object containing either the size or an error.
     *
     * @since 5.12.0
     */
    result<std::size_t>
    get_size() const noexcept;

    /**
     * Retrieve the packet_count statistic.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the packet_count or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t>
    get_packet_count() const noexcept;

    /**
     * Retrieve the current number of packets that were back-pressured in the
     * accelerator due to lack of space in the accelerator to CPU buffer.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the backpressure or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t>
    get_backpressure() const noexcept;

    /**
     * Retrieve the current number of packets dropped by the accelerator
     * because the accelerator to CPU buffer was full.
     *
     * @note Packets are dropped by the accelerator *only* if drop mode is
     * enabled for this device.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the fifo_errors or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t>
    get_fifo_errors() const noexcept;

    /**
     * Retrieve the current number of packets dropped by the accelerator
     * because the packet was bigger than the device MTU.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the truncated_count or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t>
    get_truncated_count() const noexcept;

    /**
     * Retrieve the current number of ingress packets with the error bit set
     * to 1.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the errors or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t>
    get_errors() const noexcept;

private:
    accelerator accelerator_;
    std::shared_ptr<enyx_hw_a2c_stream_descriptor const> descriptor_;
};

/**
 * @copydoc enyx_hw_c2a_stream_descriptor
 */
class c2a_stream_descriptor final
{
public:
    /**
     * Retrieve the c2a_stream's properties
     *
     * @return The c2a_stream associated properties.
     *
     * @since 5.0.0
     */
    const properties
    get_properties() const;

    /**
     * Construct a descriptor from a C descriptor.
     *
     * @param owner The owner accelerator.
     * @param descriptor The C descriptor to wrap.
     * @throws std::system_error if descriptor is nullptr.
     *
     * @since 5.0.0
     */
    c2a_stream_descriptor(accelerator const& owner,
                          enyx_hw_c2a_stream_descriptor const * descriptor);

    /**
     * Direct access to the underlying C c2a_stream_descriptor object.
     *
     * @return The C c2a_stream_descriptor object.
     *
     * @since 5.0.0
     */
    enyx_hw_c2a_stream_descriptor const *
    handle() const noexcept;

    /**
     * Retrieve the related accelerator.
     *
     * @return A reference to the accelerator.
     *
     * @since 5.5.0
     */
    accelerator const&
    get_accelerator() const noexcept;

    /**
     * Retrieve the channel MTU
     *
     * @return A result object containing either the mtu or an error.
     *
     * @since 5.11.0
     */
    result<std::uint32_t>
    get_mtu() const noexcept;

    /**
     * Retrieve the channel size
     *
     * @return A result object containing either the size or an error.
     *
     * @since 5.11.0
     */
    result<std::size_t>
    get_size() const noexcept;

private:
    accelerator accelerator_;
    std::shared_ptr<enyx_hw_c2a_stream_descriptor const> descriptor_;
};

ENYX_HW_NAMESPACE_END

#include <enyx/hw/accelerator.ipp>
