#include <boost/program_options.hpp>
#include <stdexcept>

#include "../lib/DeviceDiscovery.hpp"
#include "enyx/cores/data_stream/hw_sink.hpp"

#include <enyx/cores/tcp/emi_control.hpp>
#include <enyx/hw/c2a_stream.hpp>

namespace po = boost::program_options;

struct run_params {
    std::string accelerator_arg = "0";
    uint32_t stack_index = 0;
    uint32_t session_id = 0;
    ssize_t window_size = -1;
};

void
run_emi_command(run_params const & params)
{
    auto accelerator = enyx::create_accelerator(params.accelerator_arg);
    auto emi_hw_stream = enyx::get_tcp_events_sink(accelerator, params.stack_index);
    auto emi_sink = enyx::data_stream::sink_from_enyx_hw_stream(emi_hw_stream).v();

    auto emi_control = enyx::tcp::emi_control{emi_sink};
    if (params.window_size != -1) {
        emi_control.set_window_size(params.session_id, params.window_size).v();
        std::cout << "TCP session " << params.session_id
                  << " window size set to " << params.window_size
                  << std::endl;
    }
}

run_params
handle_options(po::variables_map args)
{
    run_params params;

    if (! args.count("accelerator")) {
        throw std::runtime_error("--accelerator option is mandatory");
    }
    params.accelerator_arg = args["accelerator"].as<std::string>();

    if (args.count("stack"))
        params.stack_index = args["stack"].as<uint32_t>();
    else
        std::cout << "Auto selected TCP stack 0" << std::endl;

    if (args.count("window-size")) {
        if (! args.count("session"))
            throw std::runtime_error{
                "--session option is mandatory with --window-size"};

        params.session_id = args["session"].as<uint32_t>();
        params.window_size = args["window-size"].as<uint32_t>();
    }

    return params;
}

po::variables_map
parse(int ac, char* av[])
{
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help,h", "produce help message")
        ("accelerator,a", po::value<std::string>(), "accelerator index")
        ("stack,s", po::value<uint32_t>(), "TCP stack index")
        ("session", po::value<uint32_t>(), "TCP session ID")
        ("window-size", po::value<uint32_t>(), "Set TCP session window size")
        ;

    po::variables_map args;

    po::store(po::parse_command_line(ac, av, desc), args);
    po::notify(args);

    if (args.count("help")) {
        std::cout << desc << std::endl;
        exit(0);
    }

    return args;
}

int main(int ac, char* av[])
{
    auto args = parse(ac, av);
    auto params = handle_options(args);
    run_emi_command(params);

    return 0;
}
