#pragma once

#include <cstddef>
#include <cstdint>

#include <atomic>
#include <chrono>
#include <future>
#include <vector>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw/c2a_stream.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/core_tree.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/udp/udp.hpp>

#include "Application.hpp"
#include "ApplicationConfiguration.hpp"
#include "BandwidthThrottle.hpp"
#include "Buffer.hpp"
#include "CacheLine.hpp"
#include "UdpSink.hpp"
#include "UdpSource.hpp"

ENYX_CORES_NAMESPACE_BEGIN

struct UdpRuntime final
{
    explicit
    UdpRuntime(ApplicationConfiguration const & configuration,
               hw::accelerator & accelerator, hw::core_tree & core_tree);

    struct Session
    {
        SessionConfiguration const * configuration;

        enum State : std::uint8_t { UNUSED, CONFIGURED };
        // This attribute is touched by tx & rx thread
        std::atomic<State> rx_state;
        // This attribute is touched by tx & rx thread
        std::atomic<State> tx_state;

        bool connect_on_receive;

        // This attribute is touched by the tx thread
        alignas(CACHE_LINE_SIZE) std::size_t sent_bytes;

        // This attribute is touched by the rx thread
        alignas(CACHE_LINE_SIZE) std::size_t received_bytes;

        std::size_t sink_backpressure;

        BandwidthThrottle bandwidth_throttle;
        std::chrono::steady_clock::time_point begin;

        // This attribute is touched by rx and tx thread.
        std::atomic<std::uint16_t> source_port;

        TxState get_tx_state();

        std::size_t get_payload_max_size() const;
    };

    using Sessions = std::vector<Session>;

    void
    create_monitoring_tasks(Tasks & tasks);

    void
    open_session(std::uint32_t session_id,
                 Session & session);

    void
    connect_session(std::uint32_t session_id,
                    Session & session);

    void
    listen_on_session(std::uint32_t session_id,
                      Session & session);

    void
    multicast_listen_on_session(std::uint32_t session_id,
                                Session & session);

    void
    close_session(std::uint32_t session_id);

    template<typename Handle>
    int
    poll_source(std::size_t source_id, Handle & handle);

    bool
    is_tx_buffer_full(Session & session,
                      std::uint32_t size);

    std::vector<UdpSource> data_sources;
    std::vector<UdpSink> data_sinks;
    udp::udp stack;
    std::uint32_t session_count;
    Sessions sessions;
    std::atomic<size_t> open_sessions_count;
};

template<typename Handle>
inline int
UdpRuntime::poll_source(std::size_t source_id, Handle & handle)
{
    auto udp_handle = [&] (std::uint32_t session_id,
                           std::uint32_t remote_ip,
                           std::uint16_t remote_port,
                           std::uint8_t const * data,
                           size_t size)
    {
        auto & session = sessions.at(session_id);
        auto const tx_state = session.tx_state.load(std::memory_order_acquire);
        if (tx_state != Session::CONFIGURED && session.connect_on_receive) {
            udp::tx_session_parameters params{};
            params.interface_id = session.configuration->interface_id;
            params.source_port = session.source_port.load(std::memory_order_acquire);
            stack.get_tx_session(session_id).value()
                 .connect(remote_ip, remote_port, params).value();
            session.tx_state.store(Session::CONFIGURED, std::memory_order_release);
        }
        handle(session_id, data, size);
    };

    return data_sources[source_id].poll_once(udp_handle);
}

ENYX_CORES_NAMESPACE_END
