#include "LinkScope.hpp"

#include <sstream>
#include <stdexcept>
#include <iomanip>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace LinkScope {

namespace en = enyx;
namespace ei = en::net_interface;

namespace {

char const USAGE[] =
"Usage: " BIN_NAME " interface link show [ IF_INDEX ]\n"
"       " BIN_NAME " interface link set IF_INDEX address MAC_ADDR\n"
"       " BIN_NAME " interface link set IF_INDEX vlan VLAN_ID [ VLAN_PCP [ VLAN_DEI ] ]\n"
"VLAN_PCP := { best_effort | background | excellent_effort |\n"
"              critical_application | video | voice |\n"
"              internetwork_control | network_control }\n"
"VLAN_DEI := { true | false }\n";

void
set_address(CmdLineIterator i, CmdLineIterator e,
            en::net_interface::net_interfaces & net_interfaces,
            int index)
{
    if (i == e)
        throw std::runtime_error{"Expected MAC_ADDR"};

    en::mac_address const mac_addr{*i};

    ensure_end(++i, e);

    net_interfaces.set_mac(index, mac_addr).value();
}

void
set_vlan(CmdLineIterator i, CmdLineIterator e,
         en::net_interface::net_interfaces & net_interfaces,
         int index)
{
    if (i == e)
        throw std::runtime_error{"Expected VLAN_ID"};

    ei::vlan_config::id const vlan_index = std::atoi(*i++);

    if (i == e)
    {
        ei::vlan_config const config{vlan_index};
        net_interfaces.set_vlan(index, config).v();
    }
    else
    {
        ei::vlan_pcp vlan_pcp{};
        std::istringstream{*i++} >> vlan_pcp;

        if (i == e)
        {
            ei::vlan_config const config{vlan_index, vlan_pcp};
            net_interfaces.set_vlan(index, config).v();
        }
        else
        {
            ei::vlan_config::dei vlan_dei{};
            std::istringstream{*i} >> std::boolalpha >> vlan_dei;

            ensure_end(++i, e);

            ei::vlan_config const vlan_config{vlan_index, vlan_pcp, vlan_dei};
            net_interfaces.set_vlan(index, vlan_config).v();
        }
    }
}

void
set(CmdLineIterator i, CmdLineIterator e,
    en::net_interface::net_interfaces & net_interfaces)
{
    if (i == e)
        throw std::runtime_error{"Expected IF_INDEX"};

    auto const index = std::atoi(*i++);

    if (i == e)
        throw std::runtime_error{"Expected { address | vlan }"};

    if (is_prefix(*i, "address"))
        set_address(++i, e, net_interfaces, index);
    else if (is_prefix(*i, "vlan"))
        set_vlan(++i, e, net_interfaces, index);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
              << "', try '" BIN_NAME " interface link help'";
        throw std::runtime_error{error.str()};
    }
}

void
show(en::net_interface::net_interfaces & net_interfaces,
     int index)
{
    std::cout << Color::IFNAME << index << Color::NONE << ": ";
    if (! net_interfaces.configured(index))
        std::cout << Color::UNCONFIGURED << "<UNCONFIGURED>" << Color::NONE;
    else
        std::cout << Color::CONFIGURED << "<CONFIGURED>" << Color::NONE;

    auto const mac = net_interfaces.get_mac(index);
    if (! mac)
    {
        std::ostringstream error;
        error << "Failed to retrieve MAC from interface "
              << index;
        throw std::system_error{mac.error(), error.str()};
    }

    std::cout << "\n\tlink " << Color::MAC << mac.value() << Color::NONE;

    auto const vlan_config = net_interfaces.get_vlan(index);
    if (! vlan_config)
    {
        std::ostringstream error;
        error << "Failed to retrieve VLAN from interface "
              << index;
        throw std::system_error{mac.error(), error.str()};
    }

    if (vlan_config.value().get_id() != 0)
        std::cout << "\n\tvlan " << vlan_config.value();

    std::cout << std::endl;
}

void
show(CmdLineIterator i, CmdLineIterator e,
     en::net_interface::net_interfaces & net_interfaces)
{
    if (i == e)
    {
        for (std::uint16_t index = 0, e = net_interfaces.count();
                index != e; ++index)
            show(net_interfaces, index);
    }
    else
    {
        auto const index = std::atoi(*i);

        ensure_end(++i, e);

        show(net_interfaces, index);
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    en::net_interface::net_interfaces & net_interfaces)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, net_interfaces);
    else if (is_prefix(*i, "set"))
        set(++i, e, net_interfaces);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i
              << "', try '" BIN_NAME " interface link help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace LinkScope

ENYX_CORES_NAMESPACE_END
