#include "IpScope.hpp"

#include <cstdlib>
#include <cstring>

#include <sstream>
#include <stdexcept>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace IpScope {

namespace en = enyx;
namespace ei = en::net_interface;

namespace {

char const USAGE[] =
"Usage: " BIN_NAME " interface ip show [ IF_INDEX ]\n"
"       " BIN_NAME " interface ip set IF_INDEX gateway IP\n"
"       " BIN_NAME " interface ip set IF_INDEX address IP_ADDRESS\n"
"IP_ADDRESS := IP/PREFIX\n";

void
show(enyx::net_interface::net_interfaces & net_interfaces,
     int index)
{
    std::cout << Color::IFNAME << index << Color::NONE << ": ";

    if (! net_interfaces.configured(index))
        std::cout << Color::UNCONFIGURED << "<UNCONFIGURED>" << Color::NONE;
    else
        std::cout << Color::CONFIGURED << "<CONFIGURED>" << Color::NONE;

    // Retrieve the IP configuration
    auto const ip_config = net_interfaces.get_ip_config(index);
    if (! ip_config)
    {
        std::ostringstream error;
        error << "Failed to retrieve IP configuration from interface "
              << index;
        throw std::system_error{ip_config.error(), error.str()};
    }

    std::cout << "\n\tinet " << Color::INET << ip_config.value().get_ip()
              << Color::NONE << "/" << int(ip_config.value().get_prefix());

    // Retrieve the gateway configuration
    auto const gateway = net_interfaces.get_gateway(index);
    if (! gateway)
    {
        std::ostringstream error;
        error << "Failed to retrieve gateway configuration from interface "
              << index;
        throw std::system_error{gateway.error(), error.str()};
    }

    // Check if gateway is set
    if (gateway.value() != en::ipv4_address{0, 0, 0, 0})
        std::cout << "\n\tgateway " << Color::INET << gateway.value()
                  << Color::NONE;

    std::cout << std::endl;
}

void
show(CmdLineIterator i, CmdLineIterator e,
     en::net_interface::net_interfaces & net_interfaces)
{
    if (i == e)
    {
        for (std::uint16_t index = 0, e = net_interfaces.count();
                index != e; ++index)
            show(net_interfaces, index);
    }
    else
    {
        auto const index = std::atoi(*i);

        ensure_end(++i, e);

        show(net_interfaces, index);
    }
}

void
set_gateway(CmdLineIterator i, CmdLineIterator e,
            en::net_interface::net_interfaces & net_interfaces,
            int index)
{
    if (i == e)
        throw std::runtime_error{"Expected IP"};

    en::ipv4_address gateway;
    std::istringstream{*i} >> gateway;

    ensure_end(++i, e);

    net_interfaces.set_gateway(index, gateway).value();
}

void
set_address(CmdLineIterator i, CmdLineIterator e,
            en::net_interface::net_interfaces & net_interfaces,
            int index)
{
    if (i == e)
        throw std::runtime_error{"Expected IP_ADDRESS"};

    // Find the ip / prefix separator
    auto sep = std::strchr(*i, '/');
    if (! sep)
    {
        std::ostringstream error;
        error << "Invalid IP_ADDRESS (" << *i << "), missing '/'";
        throw std::runtime_error{error.str()};
    }

    en::ipv4_address const ip{std::string{*i, sep}};

    // Skip the separator
    ++ sep;

    // The next string is the prefix
    char *endptr;
    std::uint8_t prefix = std::strtol(sep, &endptr, 10);
    if (*endptr != '\0')
    {
        std::ostringstream error;
        error << "Invalid IP_ADDRESS (" << *i << "), invalid prefix";
        throw std::runtime_error{error.str()};
    }

    ensure_end(++i, e);

    net_interfaces.set_ip_config(index, {ip, prefix}).value();
}

void
set(CmdLineIterator i, CmdLineIterator e,
    en::net_interface::net_interfaces & net_interfaces)
{
    if (i == e)
        throw std::runtime_error{"Expected IF_INDEX"};

    auto const index = std::atoi(*i++);

    if (i == e)
        throw std::runtime_error{"Expected { address | vlan }"};

    if (is_prefix(*i, "gateway"))
        set_gateway(++i, e, net_interfaces, index);
    else if (is_prefix(*i, "address"))
        set_address(++i, e, net_interfaces, index);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
              << "', try '" BIN_NAME " interface ip help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    enyx::net_interface::net_interfaces & net_interfaces)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, net_interfaces);
    else if (is_prefix(*i, "set"))
        set(++i, e, net_interfaces);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i
              << "', try '" BIN_NAME " interface ip help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace IpScope

ENYX_CORES_NAMESPACE_END
