/// @file net_interface.h
#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/types.h>
#include <enyx/cores_c/symbol_visibility.h>

/**
 * Interface subsystem object.
 *
 * This object represents the net_interface subsystem of a hardware network stack.
 *
 * It provides access to the configuration of all virtual net_interfaces of the
 * network stack.
 *
 * Usage:
 * @code
 * int result = -1;
 * enyx_net_interfaces * net_interfaces;
 *
 * // Create the net_interface object
 * net_interfaces = enyx_net_interfaces_create(tcp_core);
 * if (! net_interface)
 * {
 *     perror("enyx_net_interfaces_create");
 *     goto create_failed;
 * }
 *
 * // Print the of virtual net_interfaces
 * printf("Virtual net_interfaces count is  %" PRIu16 "\n",
 *        enyx_net_interfaces_count(net_interfaces));
 *
 *
 * // Configure the mac on the 3rd inteface
 * enyx_mac_address const mac;
 * if (enyx_parse_mac_address("11:22:33:44:55:66", &mac) < 0)
 * {
 *     perror("enyx_parse_mac_address");
 *     goto parse_mac_failed;
 * };
 *
 * if (enyx_net_interfaces_set_mac(net_interfaces, 2, mac) < 0)
 * {
 *     perror("enyx_net_interfaces_set_mac");
 *     goto set_mac_failed;
 * }
 *
 * result = 0;
 *
 * set_mac_failed:
 * parse_mac_failed:
 * enyx_net_interfaces_destroy(net_interfaces);
 * create_failed:
 * return result;
 * @endcode
 */
typedef struct enyx_net_interfaces enyx_net_interfaces;

/**
 * Construct the NET_INTERFACE module
 *
 * This module configures the following hw core:
 *
 * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 * * tcp_multi_stack (from TCP_STD 2.x)
 * * udp_multi_stack (from UDP_STD 2.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible enyx_hw_core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_net_interfaces *
enyx_net_interfaces_create(enyx_hw_core * stack);

/**
 * Destroy an net_interfaces subsystem.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_net_interfaces_destroy(enyx_net_interfaces * net_interfaces);

/**
 * Get an @p net_interfaces subsystem MTU (aka Ethernet MTU).
 *
 * @param net_interfaces The net_interfaces subsystem
 * @param mtu The MTU
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_get_mtu(enyx_net_interfaces const * net_interfaces,
                            uint16_t * mtu);

/**
 * Count the number of virtual net_interfaces available to a subsystem
 *
 * @param net_interfaces The net_interfaces subsystem.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint16_t
enyx_net_interfaces_count(enyx_net_interfaces const * net_interfaces);

/**
 * Test if an net_interface is configured.
 *
 * @note tcp_multi_stack and udp_multi_stack net_interfaces are configured when
 * their mac address is set.
 *
 * @param net_interfaces The net_interfaces subsystem
 * @param interface_id The net_interface id of the selected net_interface.
 * @return True if the net_interface is up, false otherwise.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_net_interfaces_configured(enyx_net_interfaces const * net_interfaces,
                               uint16_t interface_id);

/**
 * Mark an net_interface as configured. While this is a hardware setting, it is
 * mostly an indication that the net_interface should not be modified afterwards.
 *
 * @note On tcp_multi_stack and udp_multi_stack, this function is not
 * implemented.
 *
 * @param net_interfaces The net_interfaces subsystem
 * @param interface_id The net_interface id of the selected net_interface.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_configure(enyx_net_interfaces * net_interfaces,
                              uint16_t interface_id);

/**
 * Unmark an net_interface as configured. While this is a hardware setting, it is
 * mostly an indication that the net_interface may be modified afterwards.
 *
 * @note On tcp_multi_stack and udp_multi_stack, this resets the net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem
 * @param interface_id The net_interface id of the selected net_interface.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_unconfigure(enyx_net_interfaces * net_interfaces,
                                uint16_t interface_id);

/**
 * Set the mac address of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param mac The new mac address to be set.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_set_mac(enyx_net_interfaces * net_interfaces,
                            uint16_t interface_id,
                            enyx_mac_address const* mac);

/**
 * Get the mac address of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param mac The returned mac address.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_get_mac(enyx_net_interfaces const * net_interfaces,
                            uint16_t interface_id,
                            enyx_mac_address * mac);

/**
 * Define an IPv4 configuration for an net_interface.
 */
typedef struct {
    /// IPv4 Address
    enyx_ipv4_address ip;
    /// IPv4 Mask
    enyx_ipv4_address mask;
} enyx_net_interface_ip_config;

/**
 * Print the address configuration @p c into @p buffer of @p buffer_capacity
 *
 * @note At most @p buffer_capacity is written to the buffer, even
 *       if the actual string representation is longer.
 *
 * @param c The address configuration to print
 * @param buffer The buffer to fill with the string representation
 * @param buffer_capacity The capacity of the buffer
 * @return The number of character of the string representation on success,
 *         -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_print_ip_config(enyx_net_interface_ip_config const* c,
                                    char * buffer,
                                    size_t buffer_capacity);

/**
 * Set the IPv4 address of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param ip The new address to be set.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_set_ip_config(enyx_net_interfaces * net_interfaces,
                                  uint16_t interface_id,
                                  enyx_net_interface_ip_config const * ip);

/**
 * Get the IPv4 address of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param ip The returned address.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_get_ip_config(enyx_net_interfaces const * net_interfaces,
                                  uint16_t interface_id,
                                  enyx_net_interface_ip_config * ip);

/**
 * Set the IPv4 gateway of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param address The new address to be set.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_set_gateway(enyx_net_interfaces * net_interfaces,
                                uint16_t interface_id,
                                enyx_ipv4_address const * address);

/**
 * Get the IPv4 gateway of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param address The returned address.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_get_gateway(enyx_net_interfaces const * net_interfaces,
                                uint16_t interface_id,
                                enyx_ipv4_address * address);

/**
 * 802.1Q VLAN Priority code point values.
 */
typedef enum {
    /** BE - Best Effort */
    ENYX_INTERFACE_VLAN_PCP_BEST_EFFORT = 0,
    /** BK - Background */
    ENYX_INTERFACE_VLAN_PCP_BACKGROUND = 1,
    /** EE - Excellent Effort */
    ENYX_INTERFACE_VLAN_PCP_EXCELLENT_EFFORT = 2,
    /** CA - Critical Applications */
    ENYX_INTERFACE_VLAN_PCP_CRITICAL_APPLICATION = 3,
    /** VI = Video, < 100 ms latency and jitter */
    ENYX_INTERFACE_VLAN_PCP_VIDEO = 4,
    /** VO - Voice, < 10 ms latency and jitter */
    ENYX_INTERFACE_VLAN_PCP_VOICE = 5,
    /** IC - Internetwork Control */
    ENYX_INTERFACE_VLAN_PCP_INTERNETWORK_CONTROL = 6,
    /** NC - Network Control */
    ENYX_INTERFACE_VLAN_PCP_NETWORK_CONTROL = 7,
} enyx_net_interface_vlan_pcp;

/**
 *  Parse the @p mac address from @p string.
 *
 *  @param string The string to parse
 *  @param pcp The vlan PCP to fill
 *  @return @b 0 on success, @b -1 on failure (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_parse_vlan_pcp(char const * string,
                                   enyx_net_interface_vlan_pcp * pcp);

/**
 *  Return the string representation of a vlan @p pcp
 *
 *  @param pcp The pcp value to convert
 *  @return A pointer to the corresponding string PCP string or @b NULL
 *          on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY char const*
enyx_net_interfaces_get_vlan_pcp_string(enyx_net_interface_vlan_pcp pcp);

/**
 * 802.1Q VLAN configuration of an net_interface.
 */
typedef struct {
    /// VLAN ID of this net_interface (0 for default).
    uint16_t vlan_id;
    /// Priority Code Point for this net_interface (BEST_EFFORT for default).
    enyx_net_interface_vlan_pcp pcp;
    /// Drop Eligible Indicator (false for default).
    bool dei;
} enyx_net_interface_vlan_config;

/**
 *  Print the net_interface vlan configuration @p c into @p buffer of
 *  @p buffer_capacity.
 *
 *  @note At most @p buffer_capacity is written to the buffer, even
 *        if the actual string representation is longer.
 *
 *  @param c The configuration to print
 *  @param buffer The buffer to fill with the string representation
 *  @param buffer_capacity The capacity of the buffer
 *  @return The number of character of the string representation on success,
 *          -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_print_vlan_config(enyx_net_interface_vlan_config const* c,
                                      char * buffer,
                                      size_t buffer_capacity);

/**
 * Set the 802.1Q VLAN configuration of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param config The new vlan configuriation to be set.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_set_vlan_config(enyx_net_interfaces * net_interfaces,
                                    uint16_t interface_id,
                                    enyx_net_interface_vlan_config const * config);
/**
 * Set the 802.1Q VLAN configuration of an net_interface.
 *
 * @param net_interfaces The net_interfaces subsystem the @p net_interface belongs to.
 * @param interface_id The net_interface id of the selected net_interface.
 * @param config The returned vlan configuration.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interfaces_get_vlan_config(enyx_net_interfaces const * net_interfaces,
                                    uint16_t interface_id,
                                    enyx_net_interface_vlan_config * config);

typedef struct {
    /// Number of packets received by the ethernet core
    uint32_t etherrx_packet_count;
    /// Number of packets received by the ethernet core with CRC errors
    uint32_t etherrx_crc_error_count;
    /// Number of packets transmitted by the ethernet core
    uint32_t ethertx_packet_count;
    /// Number of errors on packets transmitted by the ethernet core
    uint32_t ethertx_error_count;
    /// Number of checksum errors on packets received by the IPv4 core
    uint32_t ipv4_rx_checksum_error_count;
} enyx_net_interface_statistics;


/// @cond
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_net_interface_get_statistics_(enyx_net_interfaces const * net_interfaces,
                                   enyx_net_interface_statistics * statistics,
                                   size_t statistics_size);
/// @endcond

/**
 * Get the net_interfaces subsystem Hardware Statistics
 *
 * @param net_interface The net_interfaces subsystem
 * @param statistics The statistics object
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
#define enyx_net_interfaces_get_statistics(net_interface, statistics) \
    enyx_net_interface_get_statistics_(net_interface, statistics, sizeof(*statistics))
