#pragma once

#include <stddef.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/hw_c/core.h>

typedef enum {
    ENYX_I2C_CLOCK_PRESCALE_100KHz = 0x03,
    ENYX_I2C_CLOCK_PRESCALE_200KHz = 0x02,
    ENYX_I2C_CLOCK_PRESCALE_300KHz = 0x01,
    ENYX_I2C_CLOCK_PRESCALE_400KHz = 0x00,
} enyx_i2c_clock_prescale;

/**
 * Main I2C management object.
 *
 * @note I2C addresses are defined on 7 bits.
 *       The maximum allowed value for addresses is 0x7f.
 */
typedef struct enyx_i2c enyx_i2c;

/**
 * I2C bus management object.
 */
typedef struct enyx_i2c_bus enyx_i2c_bus;

/**
 * Read from an I2C device.
 *
 * @param i2c_bus The I2C bus to use
 * @param address The I2C device address
 * @param data A buffer to fill with the read data
 * @param size The size to read from the device
 * @return The read size on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_i2c_bus_read(enyx_i2c_bus * i2c_bus, const uint8_t address, uint8_t * data,
                  size_t size);

/**
 * Write to an I2C device.
 *
 * @param i2c_bus The I2C bus to use
 * @param address The I2C device address
 * @param data A buffer containing the data to write
 * @param size The data buffer size
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_i2c_bus_write(enyx_i2c_bus * i2c_bus, const uint8_t address,
                   uint8_t const * data, size_t size);

/**
 * Read a register from an I2C device.
 *
 * @param i2c_bus The I2C bus to use
 * @param address The I2C device address
 * @param reg The device register address
 * @param data A buffer to fill with the register data
 * @param size The size of the read register
 * @return The read size on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_i2c_bus_read_register(enyx_i2c_bus * i2c_bus, const uint8_t address,
                           const uint8_t reg, uint8_t * data, size_t size);

/**
 * Write to an I2C device register.
 *
 * @param i2c_bus The I2C bus to use
 * @param address The I2C device address
 * @param reg The device register address
 * @param data A buffer containing the data to write to the register
 * @param size The data buffer size
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_i2c_bus_write_register(enyx_i2c_bus * i2c_bus, const uint8_t address,
                            const uint8_t reg, uint8_t const * data, size_t size);

/**
 * Set the I2C bus clock prescale.
 *
 * @param i2c_bus The I2C bus to use
 * @param prescale The clock prescale
 * @return 0 in case of success or -1 in case of error (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_i2c_bus_set_clock_prescale(enyx_i2c_bus * i2c_bus,
                                enyx_i2c_clock_prescale prescale);

/**
 * Set the I2C read timeout.
 *
 * @param i2c_bus The I2C bus to use
 * @param timeout_ms The timeout is miliseconds
 * @return 0 on success, -1 on error (with @b errno set accordingly)
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_i2c_bus_set_read_timeout(enyx_i2c_bus * i2c_bus, uint64_t timeout_ms);

/**
 * Destroy an I2C bus object.
 *
 * @param bus The I2C bus to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_i2c_bus_destroy(enyx_i2c_bus * bus);

/**
 * Get the I2C bus count.
 *
 * @param i2c The I2C object to use
 * @return The bus count in case of success or -1 in case of error (with
 *         @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_i2c_get_bus_count(enyx_i2c * i2c);

/**
 * Get an I2C bus from its ID.
 *
 * @param i2c The I2C object to use
 * @param bus_id The bus ID
 * @return The I2C bus object in case of success, or NULL on error (with
 *         @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_i2c_bus *
enyx_i2c_get_bus(enyx_i2c * i2c, uint8_t bus_id);

/**
 * Create an I2C object.
 *
 * @param core The root core
 * @return The I2C object on success, or NULL on error (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_i2c *
enyx_i2c_create(enyx_hw_core * core);

/**
 * Destroy an I2C subsystem.
 *
 * @param i2c The I2C to destroy
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_i2c_destroy(enyx_i2c * i2c);
