/// @file arp.h
#pragma once

#include <stdbool.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/types.h>
#include <enyx/cores_c/symbol_visibility.h>

typedef struct enyx_arp enyx_arp;

/**
 * Construct the ARP module
 *
 * This module configures the following hw core:
 *
 * * nxtcp_ip_10g_ull (from TCP_ULL 3.x)
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 * * tcp_multi_stack (from TCP_STD 2.x)
 * * udp_multi_stack (from UDP_STD 2.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible hw::core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_arp *
enyx_arp_create(enyx_hw_core * stack);

/**
 * Destroy an ARP subsystem.
 *
 * @param arp The ARP subsystem to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_arp_destroy(enyx_arp * arp);

/**
 * Check if the ARP server feature is available.
 *
 * @param arp The ARP subsystem.
 * @return true if it is available.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_arp_server_available(enyx_arp * arp);

/**
 * Check if the ARP client feature is available.
 *
 * @param arp The ARP subsystem.
 * @return true if it is available.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_arp_client_available(enyx_arp * arp);

/**
 * Check if the ARP server is enabled.
 *
 * @note Only available if @ref enyx_arp_server_available is true.
 *
 * @param arp The ARP subsystem.
 * @return true if the ARP server is enabled.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_arp_server_enabled(enyx_arp const * arp);

/**
 * Enable the ARP server of this subsystem.
 *
 * @note Only available if @ref enyx_arp_server_available is true.
 *
 * @param arp The subsystem to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_server_enable(enyx_arp * arp);

/**
 * Enable the ARP server of this subsystem.
 *
 * @note Only available if @ref enyx_arp_server_available is true.
 *
 * @param arp The subsystem to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_server_disable(enyx_arp * arp);

/**
 * Get the current timeout (in miliseconds) for ARP replies.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param timeout_ms The timeout to get.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_get_client_timeout(enyx_arp const * arp, uint16_t * timeout_ms);

/**
 * Set the current timeout (in miliseconds) for ARP replies.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param timeout_ms The timeout to set.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_set_client_timeout(enyx_arp * arp, uint16_t timeout_ms);

/**
 * Get the current lifetime (in seconds) of dynamic ARP entries.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param lifetime_s The lifetime to get.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_get_dynamic_lifetime(enyx_arp const * arp,
                                        uint16_t * lifetime_s);

/**
 * Set the current lifetime (in seconds) of dynamic ARP entries.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param lifetime_s The lifetime to set.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_set_dynamic_lifetime(enyx_arp * arp,
                                        uint16_t lifetime_s);

/**
 * Get the maximum number of entries in the ARP table.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @return The maximum number of entry
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint16_t
enyx_arp_table_get_max_entry(enyx_arp const * arp);

/**
 * Get the mac address associated with an IP address in the ARP table.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param ipv4 The IP address
 * @param mac The mac address
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_find_mac(enyx_arp * arp,
                            enyx_ipv4_address const * ipv4,
                            enyx_mac_address * mac);

/**
 * Get an entry from the ARP table.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param entry_id The index of the required entry.
 * @param ipv4 The IPv4 address of the entry
 * @param mac The MAC address of the entry
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_get_entry(enyx_arp const * arp,
                             uint16_t entry_id,
                             enyx_ipv4_address * ipv4,
                             enyx_mac_address * mac);

/**
 * Add a new ARP entry to the ARP table.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem.
 * @param ipv4 The ipv4 address to add
 * @param mac The mac address to use
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_add_entry(enyx_arp * arp,
                             enyx_ipv4_address const * ipv4,
                             enyx_mac_address const * mac);

/**
 * Delete an entry from the ARP table.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @param ipv4 The IPv4 identifying the entry.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_delete_entry(enyx_arp * arp,
                                enyx_ipv4_address const * ipv4);

/**
 * Delete all dynamic entries from the ARP table.
 *
 * @note Only available if @ref enyx_arp_client_available is true.
 *
 * @param arp The ARP subsystem
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_arp_table_flush_dynamic_entries(enyx_arp * arp);
