#pragma once

#include <enyx/cores_c/udp/udp.h>

#include <vector>
#include <memory>

#include <enyx/hw/core.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/udp/session.hpp>
#include <enyx/cores/result.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_udp>
{
    void
    operator()(::enyx_udp * ptr) const
    {
        ::enyx_udp_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN
namespace udp {

/// @copydoc enyx_udp_statistics
using statistics = enyx_udp_statistics;

/**
 * @copydoc enyx_udp
 */
class udp
{
public:
    /**
     * Construct the UDP module
     *
     * This module configures the following hw::core:
     *
     * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
     * * udp_multi_stack (from UDP_STD 2.x)
     *
     * @param udp_core The hardware core containing the stack.
     * @throw system_error on failure (unsupported hardware).
     *
     * @note This requires you to find a compatible hw::core core using the
     *       enyx-hw library (e.g. using enyx::hw::core::enumerate).
     */
    udp(enyx::hw::core const& udp_core);

    /**
     *  Retrieve the associated core.
     *
     *  @return The core.
     */
    enyx::hw::core
    get_core() const noexcept;

    /**
     * Get the name of this stack.
     *
     * @return A result object containing the stack name on success or an
     *         error on failure.
     */
    result<std::string>
    get_name() const noexcept;

    /**
     * Get the MTU of this stack.
     *
     * @return A result object containing the MTU on success or an error on
     *         failure.
     */
    result<uint16_t>
    get_mtu() const noexcept;

    /**
     * Is safe mode enabled on this udp subsystem.
     *
     * @note If the underlying hardware core is a udp_multi_stack this parameter is
     * not persistent in the hardware and must be set once for each run of your
     * program (default is disabled).
     *
     * @return true if safe mode is enabled, false otherwise.
     */
    bool
    is_safe_mode_enabled() const noexcept;

    /**
     * Enable safe mode on this subsystem.
     *
     * @note If the underlying hardware core is a udp_multi_stack this parameter is
     * not persistent in the hardware and must be set once for each run of your
     * program (default is disabled).
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_safe_mode() noexcept;

    /**
     * Disable safe mode on this subsystem.
     *
     * @note If the underlying hardware core is a udp_multi_stack this parameter is
     * not persistent in the hardware and must be set once for each run of your
     * program (default is disabled).
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_safe_mode() noexcept;

    /**
     * Is checksum verification enabled on this udp subsystem.
     *
     * @note This feature is not supported by the udp_multi_stack hardware core.
     *
     * @return A result object containing the checksum verification activation
     *         state on success or an error on failure.
     */
    result<bool>
    is_checksum_verification_enabled() const noexcept;

    /**
     * Enable checksum verification on this subsystem.
     *
     * @note This feature is not supported by the udp_multi_stack hardware core.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    enable_checksum_verification() noexcept;

    /**
     * Disable checksum verification on this subsystem.
     *
     * @note This feature is not supported by the udp_multi_stack hardware core.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    disable_checksum_verification() noexcept;

    /**
     * Get the number of rx sessions supported by this UDP subsystem.
     *
     * @return The number of rx sessions.
     */
    std::uint16_t
    get_rx_session_count() const noexcept;

    /**
     * Get a rx_session object from this UDP subsystem.
     *
     * @param session_id The id of the requested rx session.
     *
     * @return A result object containing the rx session on success or an error
     *         on failure.
     */
    result<rx_session>
    get_rx_session(std::uint16_t session_id) const noexcept;

    /**
     * @copybrief enyx_udp_get_available_rx_session
     *
     * @return A result object containing a session on success or an error
     *         on failure.
     */
    result<rx_session>
    get_available_rx_session() const noexcept;

    /**
     * Get the number of tx sessions supported by this UDP subsystem.
     *
     * @return The number of tx sessions.
     */
    std::uint16_t
    get_tx_session_count() const noexcept;

    /**
     * Get a tx session object from this UDP subsystem.
     *
     * @param session_id The id of the requested tx session.
     *
     * @return A result object containing the tx session on success or an error
     *         on failure.
     */
    result<tx_session>
    get_tx_session(std::uint16_t session_id) const noexcept;

    /**
     * @copybrief enyx_udp_get_available_rx_session
     *
     * @return A result object containing a session on success or an error
     *         on failure.
     */
    result<tx_session>
    get_available_tx_session() const noexcept;

    /**
     * @copybrief enyx_udp_get_statistics
     *
     * @return A result object containing the statistics on success or an error
     *         on failure.
     */
    result<statistics>
    get_statistics() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_udp *
    handle() const noexcept;

private:
    enyx::hw::core udp_core_;
    std::shared_ptr<::enyx_udp> udp_;
};

} /* namespace udp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/udp/udp.ipp>
