#include <cerrno>

#include <algorithm>
#include <iostream>
#include <iterator>
#include <sstream>
#include <system_error>

ENYX_CORES_NAMESPACE_BEGIN

inline
mac_address::mac_address() noexcept
    : address_()
{ }

inline
mac_address::mac_address(enyx_mac_address const& c_address) noexcept
    : address_(c_address)
{ }

inline
mac_address::mac_address(std::string const& str)
{
    if (::enyx_parse_mac_address(str.c_str(), &address_) < 0)
        throw std::system_error{errno, std::generic_category(),
                                "mac_address::mac_address()"};
}

inline
mac_address::mac_address(std::uint64_t integer)
{
    if (::enyx_mac_address_from_int(integer, &address_) < 0)
        throw std::system_error{errno, std::generic_category(),
                                "mac_address::mac_address()"};
}

inline
mac_address::mac_address(std::initializer_list<std::uint8_t> l)
    : mac_address(l.begin(), l.end())
{ }

template<typename Iterator>
inline
mac_address::mac_address(Iterator begin, Iterator end)
{
    if (std::distance(begin, end) != sizeof(address_.value))
        throw std::system_error{EINVAL, std::generic_category(),
                                "mac_address::mac_address()"};

    std::copy(begin, end, address_.value);
}

inline mac_address::iterator
mac_address::begin() noexcept
{
    return std::begin(address_.value);
}

inline mac_address::iterator
mac_address::end() noexcept
{
    return std::end(address_.value);
}

inline mac_address::const_iterator
mac_address::begin() const noexcept
{
    return std::begin(address_.value);
}

inline mac_address::const_iterator
mac_address::end() const noexcept
{
    return std::end(address_.value);
}

inline
mac_address::operator ::enyx_mac_address * () noexcept
{
    return &address_;
}

inline
mac_address::operator ::enyx_mac_address const* () const noexcept
{
    return &address_;
}

inline std::uint64_t
mac_address::to_int() const noexcept
{
    return ::enyx_mac_address_to_int(&address_);
}

inline std::ostream &
operator<<(std::ostream & out, mac_address const& mac)
{
    std::ostream::sentry sentry{out};

    if (sentry)
    {
        char buffer[18];
        int const ret = ::enyx_print_mac_address(mac, buffer, sizeof(buffer));

        if (ret < 0)
            throw std::system_error{errno, std::generic_category(),
                                    "enyx_print_mac_address()"};
        if (std::size_t(ret) >= sizeof(buffer))
            throw std::logic_error{"Buffer too small for "
                                    "enyx_print_mac_address()"};

        out << buffer;
    }

    return out;
}

inline std::istream &
operator>>(std::istream & in, mac_address & mac)
{
    std::istream::sentry sentry{in};

    if (sentry)
    {
        std::string buffer;
        in >> buffer;
        mac = mac_address(buffer);
    }

    return in;
}

inline bool
operator==(mac_address const& a, mac_address const& b) noexcept
{
    return std::equal(a.begin(), a.end(), b.begin());
}

inline bool
operator!=(mac_address const& a, mac_address const& b) noexcept
{
    return ! (a == b);
}

inline
ipv4_address::ipv4_address() noexcept
    : address_()
{ }

inline
ipv4_address::ipv4_address(enyx_ipv4_address const& c_address) noexcept
    : address_(c_address)
{ }

inline
ipv4_address::ipv4_address(std::string const& str)
{
    if (::enyx_parse_ipv4_address(str.c_str(), &address_) < 0)
        throw std::system_error{errno, std::generic_category(),
                                "ipv4_address::ipv4_address()"};
}

inline
ipv4_address::ipv4_address(std::uint32_t integer)
{
    if (::enyx_ipv4_address_from_int(integer, &address_) < 0)
        throw std::system_error{errno, std::generic_category(),
                                "ipv4_address::ipv4_address()"};
}

inline
ipv4_address::ipv4_address(std::initializer_list<std::uint8_t> l)
    : ipv4_address(l.begin(), l.end())
{ }

template<typename Iterator>
inline
ipv4_address::ipv4_address(Iterator begin, Iterator end)
{
    if (std::distance(begin, end) != sizeof(address_.value))
        throw std::system_error{EINVAL, std::generic_category(),
                                "ipv4_address::ipv4_address()"};

    std::copy(begin, end, address_.value);
}

inline ipv4_address::iterator
ipv4_address::begin() noexcept
{
    return std::begin(address_.value);
}

inline ipv4_address::iterator
ipv4_address::end() noexcept
{
    return std::end(address_.value);
}

inline ipv4_address::const_iterator
ipv4_address::begin() const noexcept
{
    return std::begin(address_.value);
}

inline ipv4_address::const_iterator
ipv4_address::end() const noexcept
{
    return std::end(address_.value);
}

inline
ipv4_address::operator ::enyx_ipv4_address * () noexcept
{
    return &address_;
}

inline
ipv4_address::operator ::enyx_ipv4_address const* () const noexcept
{
    return &address_;
}

inline std::uint32_t
ipv4_address::to_int() const noexcept
{
    return ::enyx_ipv4_address_to_int(&address_);
}

inline std::ostream &
operator<<(std::ostream & out, ipv4_address const& ipv4)
{
    std::ostream::sentry sentry{out};

    if (sentry)
    {
        char buffer[16];
        int const ret = enyx_print_ipv4_address(ipv4, buffer, sizeof(buffer));

        if (ret < 0)
            throw std::system_error{errno, std::generic_category(),
                                    "enyx_print_ipv4_address()"};
        if (std::size_t(ret) >= sizeof(buffer))
            throw std::logic_error{"Buffer too small for "
                                    "enyx_print_ipv4_address()"};

        out << buffer;
    }

    return out;
}

inline std::istream &
operator>>(std::istream & in, ipv4_address & ipv4)
{
    std::istream::sentry sentry{in};

    if (sentry)
    {
        std::string buffer;
        in >> buffer;
        ipv4 = ipv4_address(buffer);
    }

    return in;
}

inline bool
operator==(ipv4_address const& a, ipv4_address const& b) noexcept
{
    return std::equal(a.begin(), a.end(), b.begin());
}

inline bool
operator!=(ipv4_address const& a, ipv4_address const& b) noexcept
{
    return ! (a == b);
}

inline
ipv4_endpoint::ipv4_endpoint() noexcept
    : endpoint_()
{ }

inline
ipv4_endpoint::ipv4_endpoint(enyx_ipv4_endpoint const& c_endpoint) noexcept
    : endpoint_(c_endpoint)
{ }

inline
ipv4_endpoint::ipv4_endpoint(ipv4_address const & address,
                             std::uint16_t port)
        : endpoint_{*address, port}
{ }

inline
ipv4_endpoint::ipv4_endpoint(std::string const& str)
{
    if (::enyx_parse_ipv4_endpoint(str.c_str(), &endpoint_) < 0)
        throw std::system_error{errno, std::generic_category(),
                                "ipv4_endpoint::ipv4_endpoint()"};
}

inline ipv4_address
ipv4_endpoint::address() const noexcept
{
    return endpoint_.address;
}

inline std::uint16_t
ipv4_endpoint::port() const noexcept
{
    return endpoint_.port;
}

inline
ipv4_endpoint::operator ::enyx_ipv4_endpoint * () noexcept
{
    return &endpoint_;
}

inline
ipv4_endpoint::operator ::enyx_ipv4_endpoint const* () const noexcept
{
    return &endpoint_;
}

inline std::ostream &
operator<<(std::ostream & out, ipv4_endpoint const& ipv4)
{
    std::ostream::sentry sentry{out};

    if (sentry)
    {
        char buffer[22];
        int const ret = ::enyx_print_ipv4_endpoint(ipv4, buffer, sizeof(buffer));

        if (ret < 0)
            throw std::system_error{errno, std::generic_category(),
                                    "enyx_print_ipv4_endpoint()"};
        if (std::size_t(ret) >= sizeof(buffer))
            throw std::logic_error{"Buffer too small for "
                                   "enyx_print_ipv4_endpoint()"};

        out << buffer;
    }

    return out;
}

inline std::istream &
operator>>(std::istream & in, ipv4_endpoint & endpoint)
{
    std::istream::sentry sentry{in};

    if (sentry)
    {
        std::string buffer;
        in >> buffer;
        endpoint = ipv4_endpoint(buffer);
    }

    return in;
}

inline bool
operator==(ipv4_endpoint const& a, ipv4_endpoint const& b) noexcept
{
    return a.address() == b.address() && a.port() == b.port();
}

inline bool
operator!=(ipv4_endpoint const& a, ipv4_endpoint const& b) noexcept
{
    return ! (a == b);
}

ENYX_CORES_NAMESPACE_END
